// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __CConstant_h__
#define __CConstant_h__

// arithmetic constants

#include "Puma/Limits.h"
#include "Puma/CTypeInfo.h"
#include "Puma/CExprValue.h"

namespace Puma {


class CConstant : public CExprValue {
  union Value {
    LONG_LONG    int_val;    // [signed] char|wchar_t|short|int|long|long long
    U_LONG_LONG  uint_val;   // bool|unsigned (char|short|int|long|long long)
    long double  float_val;  // float|double|long double
    Value () { int_val = 0; uint_val = 0; float_val = 0.0; }
  } value;
  
  enum {
    INT_VAL,
    UINT_VAL,
    FLOAT_VAL
  } value_type;
  
public:
  CConstant (LONG_LONG v, CTypeInfo *t) : CExprValue (t)
   { value.int_val = v; value_type = INT_VAL; }
  CConstant (U_LONG_LONG v, CTypeInfo *t) : CExprValue (t)
   { value.uint_val = v; value_type = UINT_VAL; }
  CConstant (long double v, CTypeInfo *t) : CExprValue (t)
   { value.float_val = v; value_type = FLOAT_VAL; }
  virtual ~CConstant () {}
   
  CConstant *cast_to (CTypeInfo *) const;

  CConstant *compute (int) const;
  CConstant *compute (int, const CConstant *) const;

  LONG_LONG convert_to_int () const;
  U_LONG_LONG convert_to_uint () const;
  long double convert_to_float () const;

  CConstant *Constant () const { return (CConstant*)this; }
  CConstant *duplicate () const;

  // compares only the values, not the types!
  bool operator== (const CConstant &) const;
  bool operator!= (const CConstant &) const;

  bool isNull () const;
  bool isPositive () const;
  bool isNegative () const;
  bool isSigned () const    { return value_type == INT_VAL; }
  bool isUnsigned () const  { return value_type == UINT_VAL; }
  bool isFloat () const     { return value_type == FLOAT_VAL; }

  // print this value
  virtual void print (ostream &out) const;
  
private:
  LONG_LONG compute (int, LONG_LONG) const;
  LONG_LONG compute (LONG_LONG, int, LONG_LONG) const;
  U_LONG_LONG compute (int, U_LONG_LONG) const;
  U_LONG_LONG compute (U_LONG_LONG, int, U_LONG_LONG) const;
  long double compute (int, long double) const;
  long double compute (long double, int, long double) const;

  LONG_LONG cast_to_real_type (LONG_LONG, CTypeInfo *) const;
  U_LONG_LONG cast_to_real_type (U_LONG_LONG, CTypeInfo *) const;
  long double cast_to_real_type (long double, CTypeInfo *) const;
};


} // namespace Puma

#endif /* __CConstant_h__ */
