/*
 * Copyright (C) 2005, 2006, 2007 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: xino.c,v 1.36 2007/07/30 04:13:07 sfjro Exp $ */

#include <linux/fsnotify.h>
#include <linux/smp_lock.h>
#include <linux/version.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 18)
#include <linux/uaccess.h>
#else
#include <asm/uaccess.h>
#endif
#include "aufs.h"

/* ---------------------------------------------------------------------- */

static ssize_t xino_fread(readf_t func, struct file *file, void *buf,
			  size_t size, loff_t *pos)
{
	ssize_t err;
	mm_segment_t oldfs;

	LKTRTrace("%.*s, sz %lu, *pos %Ld\n",
		  DLNPair(file->f_dentry), (unsigned long)size, *pos);

	oldfs = get_fs();
	set_fs(KERNEL_DS);
	do {
		// signal_pending
		err = func(file, (char __user*)buf, size, pos);
	} while (err == -EAGAIN || err == -EINTR);
	set_fs(oldfs);

#if 0
	if (err > 0)
		fsnotify_access(file->f_dentry);
#endif

	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

static ssize_t do_xino_fwrite(writef_t func, struct file *file, void *buf,
			      size_t size, loff_t *pos)
{
	ssize_t err;
	mm_segment_t oldfs;

	lockdep_off();
	oldfs = get_fs();
	set_fs(KERNEL_DS);
	do {
		// signal_pending
		err = func(file, (const char __user*)buf, size, pos);
	} while (err == -EAGAIN || err == -EINTR);
	set_fs(oldfs);
	lockdep_on();

#if 0
	if (err > 0)
		fsnotify_modify(file->f_dentry);
#endif

	TraceErr(err);
	return err;
}

struct do_xino_fwrite_args {
	ssize_t *errp;
	writef_t func;
	struct file *file;
	void *buf;
	size_t size;
	loff_t *pos;
};

static void call_do_xino_fwrite(void *args)
{
	struct do_xino_fwrite_args *a = args;
	*a->errp = do_xino_fwrite(a->func, a->file, a->buf, a->size, a->pos);
}

static ssize_t xino_fwrite(writef_t func, struct file *file, void *buf,
			   size_t size, loff_t *pos)
{
	ssize_t err;

	LKTRTrace("%.*s, sz %lu, *pos %Ld\n",
		  DLNPair(file->f_dentry), (unsigned long)size, *pos);

	// signal block and no wkq?
	/*
	 * it breaks RLIMIT_FSIZE and normal user's limit,
	 * users should care about quota and real 'filesystem full.'
	 */
	if (!is_au_wkq(current)) {
		int wkq_err;
		struct do_xino_fwrite_args args = {
			.errp	= &err,
			.func	= func,
			.file	= file,
			.buf	= buf,
			.size	= size,
			.pos	= pos
		};
		wkq_err = au_wkq_wait(call_do_xino_fwrite, &args, /*dlgt*/0);
		if (unlikely(wkq_err))
			err = wkq_err;
	} else
		err = do_xino_fwrite(func, file, buf, size, pos);

	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

#ifndef LLONG_MAX /* before linux-2.6.18 */
#define LLONG_MAX	((long long)(~0ULL >> 1))
#endif
#define Au_LOFF_MAX	((loff_t)LLONG_MAX)

/*
 * write @ino to the xinofile for the specified branch{@sb, @bindex}
 * at the position of @_ino.
 * when @ino is zero, it is written to the xinofile and means no entry.
 */
int xino_write(struct super_block *sb, aufs_bindex_t bindex, ino_t h_ino,
	       struct xino *xino)
{
	loff_t pos;
	ssize_t sz;
	struct file *file;

	LKTRTrace("b%d, hi%lu, i%lu\n", bindex, h_ino, xino->ino);
	BUILD_BUG_ON(sizeof(long long) != sizeof(Au_LOFF_MAX)
		     || ((loff_t)-1) > 0);

	if (unlikely(!au_flag_test(sb, AuFlag_XINO)))
		return 0;

	pos = h_ino;
	if (unlikely(Au_LOFF_MAX / sizeof(*xino) - 1 < pos)) {
		IOErr1("too large hi%lu\n", h_ino);
		return -EFBIG;
	}
	pos *= sizeof(*xino);
	file = stobr(sb, bindex)->br_xino;
	AuDebugOn(!file);
	sz = xino_fwrite(stosi(sb)->si_xwrite, file, xino, sizeof(*xino), &pos);
	//if (LktrCond) sz = 1;
	if (sz == sizeof(*xino))
		return 0; /* success */

	IOErr("write failed (%ld)\n", (long)sz);
	return -EIO;
}

/* ---------------------------------------------------------------------- */

static const int page_bits = (int)PAGE_SIZE * BITS_PER_BYTE;
//static const int page_bits = 4;
static ino_t xib_calc_ino(unsigned long pindex, int bit)
{
	ino_t ino;
	AuDebugOn(bit < 0 || page_bits <= bit);
	ino = AUFS_FIRST_INO + pindex * page_bits + bit;
	return ino;
}

static void xib_calc_bit(ino_t ino, unsigned long *pindex, int *bit)
{
	AuDebugOn(ino < AUFS_FIRST_INO);
	ino -= AUFS_FIRST_INO;
	*pindex = ino / page_bits;
	*bit = ino % page_bits;
}

static int xib_pindex(struct super_block *sb, unsigned long pindex)
{
	int err;
	struct aufs_sbinfo *sbinfo;
	loff_t pos;
	ssize_t sz;
	struct file *xib;
	unsigned long *p;

	LKTRTrace("pindex %lu\n", pindex);
	sbinfo = stosi(sb);
	MtxMustLock(&sbinfo->si_xib_mtx);
	AuDebugOn(pindex > ULONG_MAX / PAGE_SIZE
		  || !au_flag_test(sb, AuFlag_XINO));

	if (pindex == sbinfo->si_xib_last_pindex)
		return 0;

	xib = sbinfo->si_xib;
	p = sbinfo->si_xib_buf;
	pos = sbinfo->si_xib_last_pindex;
	pos *= PAGE_SIZE;
	sz = xino_fwrite(sbinfo->si_xwrite, xib, p, PAGE_SIZE, &pos);
	if (unlikely(sz != PAGE_SIZE))
		goto out;

	pos = pindex;
	pos *= PAGE_SIZE;
	err = au_update_fuse_h_inode(xib->f_vfsmnt, xib->f_dentry,
				     need_dlgt(sb));
	AuDebugOn(err);
	if (i_size_read(xib->f_dentry->d_inode) >= pos + PAGE_SIZE)
		sz = xino_fread(sbinfo->si_xread, xib, p, PAGE_SIZE, &pos);
	else {
		memset(p, 0, PAGE_SIZE);
		sz = xino_fwrite(sbinfo->si_xwrite, xib, p, PAGE_SIZE, &pos);
	}
	if (sz == PAGE_SIZE) {
		sbinfo->si_xib_last_pindex = pindex;
		return 0; /* success */
	}

 out:
	IOErr1("write failed (%ld)\n", (long)sz);
	err = sz;
	if (sz >= 0)
		err = -EIO;
	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

int xino_write0(struct super_block *sb, aufs_bindex_t bindex, ino_t h_ino,
		ino_t ino)
{
	int err, bit;
	unsigned long pindex;
	struct aufs_sbinfo *sbinfo;
	struct xino xino = {
		.ino	= 0
	};

	LKTRTrace("b%d, hi%lu, i%lu\n", bindex, h_ino, ino);

	if (unlikely(!au_flag_test(sb, AuFlag_XINO)))
		return 0;

	err = 0;
	if (unlikely(ino)) {
		AuDebugOn(ino < AUFS_FIRST_INO);
		xib_calc_bit(ino, &pindex, &bit);
		sbinfo = stosi(sb);
		AuDebugOn(page_bits <= bit);
		mutex_lock(&sbinfo->si_xib_mtx);
		err = xib_pindex(sb, pindex);
		if (!err) {
			clear_bit(bit, sbinfo->si_xib_buf);
			sbinfo->si_xib_next_bit = bit;
		}
		mutex_unlock(&sbinfo->si_xib_mtx);
	}

	if (!err)
		err = xino_write(sb, bindex, h_ino, &xino);
	return err;
}

ino_t xino_new_ino(struct super_block *sb)
{
	ino_t ino;
	struct aufs_sbinfo *sbinfo;
	int free_bit, err;
	unsigned long *p, pindex, ul, pend;
	struct file *file;

	//au_debug_on();
	TraceEnter();

	if (unlikely(!au_flag_test(sb, AuFlag_XINO)))
		return iunique(sb, AUFS_FIRST_INO);

	sbinfo = stosi(sb);
	mutex_lock(&sbinfo->si_xib_mtx);
	p = sbinfo->si_xib_buf;
	free_bit = sbinfo->si_xib_next_bit;
	//Dbg("bit %d, pindex %d\n", free_bit, sbinfo->si_xib_last_pindex);
	//Dbg("bit %d, %d\n", free_bit, test_bit(free_bit, p));
	if (free_bit < page_bits && !test_bit(free_bit, p)) {
		LKTRLabel(here);
		goto out; /* success */
	}
	free_bit = find_first_zero_bit(p, page_bits);
	if (free_bit < page_bits) {
		LKTRLabel(here);
		goto out; /* success */
	}

	pindex = sbinfo->si_xib_last_pindex;
	for (ul = pindex - 1; ul < ULONG_MAX; ul--) {
		err = xib_pindex(sb, ul);
		if (unlikely(err))
			goto out_err;
		free_bit = find_first_zero_bit(p, page_bits);
		if (free_bit < page_bits) {
			LKTRLabel(here);
			goto out; /* success */
		}
	}

	file = sbinfo->si_xib;
	err = au_update_fuse_h_inode(file->f_vfsmnt, file->f_dentry,
				     need_dlgt(sb));
	AuDebugOn(err);
	pend = i_size_read(file->f_dentry->d_inode) / PAGE_SIZE;
	for (ul = pindex + 1; ul <= pend; ul++) {
		err = xib_pindex(sb, ul);
		if (unlikely(err))
			goto out_err;
		free_bit = find_first_zero_bit(p, page_bits);
		if (free_bit < page_bits) {
			LKTRLabel(here);
			goto out; /* success */
		}
	}
	BUG();

 out:
	set_bit(free_bit, p);
	sbinfo->si_xib_next_bit++;
	pindex = sbinfo->si_xib_last_pindex;
	mutex_unlock(&sbinfo->si_xib_mtx);
	ino = xib_calc_ino(pindex, free_bit);
	//Dbg("i%lu\n", ino);
	LKTRTrace("i%lu\n", ino);
	//au_debug_off();
	return ino;
 out_err:
	mutex_unlock(&sbinfo->si_xib_mtx);
	LKTRTrace("i0\n");
	//au_debug_off();
	return 0;
}

/*
 * read @ino from xinofile for the specified branch{@sb, @bindex}
 * at the position of @h_ino.
 * if @ino does not exist and @do_new is true, get new one.
 */
int xino_read(struct super_block *sb, aufs_bindex_t bindex, ino_t h_ino,
	      struct xino *xino)
{
	int err;
	struct file *file;
	loff_t pos;
	ssize_t sz;

	LKTRTrace("b%d, hi%lu\n", bindex, h_ino);

	err = 0;
	xino->ino = 0;
	if (unlikely(!au_flag_test(sb, AuFlag_XINO)))
		return 0; /* no ino */

	pos = h_ino;
	if (unlikely(Au_LOFF_MAX / sizeof(*xino) - 1 < pos)) {
		IOErr1("too large hi%lu\n", h_ino);
		return -EFBIG;
	}
	pos *= sizeof(*xino);

	file = stobr(sb, bindex)->br_xino;
	AuDebugOn(!file);
	err = au_update_fuse_h_inode(file->f_vfsmnt, file->f_dentry,
				     need_dlgt(sb));
	AuDebugOn(err);
	if (i_size_read(file->f_dentry->d_inode) < pos + sizeof(*xino))
		return 0; /* no ino */

	sz = xino_fread(stosi(sb)->si_xread, file, xino, sizeof(*xino), &pos);
	if (sz == sizeof(*xino))
		return 0; /* success */

	err = sz;
	if (unlikely(sz >= 0)) {
		err = -EIO;
		IOErr("xino read error (%ld)\n", (long)sz);
	}

	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

struct file *xino_create(struct super_block *sb, char *fname, int silent,
			 struct dentry *parent)
{
	struct file *file;
	int err;
	struct dentry *h_parent;
	struct inode *h_dir;
	//const int udba = au_flag_test(sb, AuFlag_UDBA_INOTIFY);

	LKTRTrace("%s\n", fname);
	//AuDebugOn(!au_flag_test(sb, AuFlag_XINO));

	/* LSM may detect it */
	file = vfsub_filp_open(fname, O_RDWR | O_CREAT | O_EXCL | O_LARGEFILE,
			       S_IRUGO | S_IWUGO);
	//file = ERR_PTR(-1);
	if (IS_ERR(file)) {
		if (!silent)
			Err("open %s(%ld)\n", fname, PTR_ERR(file));
		return file;
	}
#if 0
	if (unlikely(udba && parent))
		au_direval_dec(parent);
#endif

	/* keep file count */
	h_parent = dget_parent(file->f_dentry);
	h_dir = h_parent->d_inode;
	vfsub_i_lock_nested(h_dir, AuLsc_I_PARENT);
	err = vfsub_unlink(h_dir, file->f_dentry, /*dlgt*/0);
#if 0
	if (unlikely(!err && udba && parent))
		au_direval_dec(parent);
#endif
	vfsub_i_unlock(h_dir);
	dput(h_parent);
	if (unlikely(err)) {
		if (!silent)
			Err("unlink %s(%d)\n", fname, err);
		goto out;
	}

	if (sb != file->f_dentry->d_sb)
		return file; /* success */

	if (!silent)
		Err("%s must be outside\n", fname);
	err = -EINVAL;

 out:
	fput(file);
	file = ERR_PTR(err);
	return file;
}

/*
 * find another branch who is on the same filesystem of the specified
 * branch{@btgt}. search until @bend.
 */
static int is_sb_shared(struct super_block *sb, aufs_bindex_t btgt,
			aufs_bindex_t bend)
{
	aufs_bindex_t bindex;
	struct super_block *tgt_sb = sbr_sb(sb, btgt);

	for (bindex = 0; bindex <= bend; bindex++)
		if (unlikely(btgt != bindex && tgt_sb == sbr_sb(sb, bindex)))
			return bindex;
	return -1;
}

/*
 * create a new xinofile at the same place/path as @base_file.
 */
static struct file *xino_create2(struct super_block *sb, struct file *base_file,
				 struct file *copy_src)
{
	struct file *file;
	int err, sparse;
	struct dentry *base, *dentry, *parent;
	struct inode *dir, *inode;
	struct qstr *name;
	struct lkup_args lkup = {
		.nfsmnt	= NULL,
		.dlgt	= 0
	};

	base = base_file->f_dentry;
	LKTRTrace("%.*s\n", DLNPair(base));
	parent = dget_parent(base);
	dir = parent->d_inode;
	IMustLock(dir);

	file = ERR_PTR(-EINVAL);
	if (unlikely(au_is_nfs(parent->d_sb)))
		goto out;

	/* do not superio, nor NFS. */
	name = &base->d_name;
	dentry = lkup_one(name->name, parent, name->len, &lkup);
	//if (LktrCond) {dput(dentry); dentry = ERR_PTR(-1);}
	if (IS_ERR(dentry)) {
		file = (void*)dentry;
		Err("%.*s lookup err %ld\n", LNPair(name), PTR_ERR(dentry));
		goto out;
	}
	err = vfsub_create(dir, dentry, S_IRUGO | S_IWUGO, NULL, /*dlgt*/0);
	//if (LktrCond) {vfs_unlink(dir, dentry); err = -1;}
	if (unlikely(err)) {
		file = ERR_PTR(err);
		Err("%.*s create err %d\n", LNPair(name), err);
		goto out_dput;
	}
	file = dentry_open(dget(dentry), mntget(base_file->f_vfsmnt),
			   O_RDWR | O_CREAT | O_EXCL | O_LARGEFILE);
	//if (LktrCond) {fput(file); file = ERR_PTR(-1);}
	if (IS_ERR(file)) {
		Err("%.*s open err %ld\n", LNPair(name), PTR_ERR(file));
		goto out_dput;
	}
	err = vfsub_unlink(dir, dentry, /*dlgt*/0);
	//if (LktrCond) err = -1;
	if (unlikely(err)) {
		Err("%.*s unlink err %d\n", LNPair(name), err);
		goto out_fput;
	}

	if (unlikely(copy_src)) {
		inode = copy_src->f_dentry->d_inode;
		err = au_update_fuse_h_inode(copy_src->f_vfsmnt,
					     copy_src->f_dentry, /*dlgt*/0);
		AuDebugOn(err);
		err = au_copy_file(file, copy_src, i_size_read(inode), sb,
				   &sparse);
		if (unlikely(err)) {
			Err("%.*s copy err %d\n", LNPair(name), err);
			goto out_fput;
		}
	}
	goto out_dput; /* success */

 out_fput:
	fput(file);
	file = ERR_PTR(err);
 out_dput:
	dput(dentry);
 out:
	dput(parent);
	TraceErrPtr(file);
	return file;
}

/* ---------------------------------------------------------------------- */

/*
 * initialize the xinofile for the specified branch{@sb, @bindex}
 * at the place/path where @base_file indicates.
 * test whether another branch is on the same filesystem or not,
 * if @do_test is true.
 */
int xino_br(struct super_block *sb, aufs_bindex_t bindex,
	    struct file *base_file, int do_test)
{
	int err, do_create;
	struct aufs_branch *br, *shared_br;
	aufs_bindex_t bshared, bend;
	struct inode *inode, *h_inode, *dir;
	struct xino xino;
	struct dentry *parent;
	struct file *file;

	LKTRTrace("b%d, base_file %p, do_test %d\n",
		  bindex, base_file, do_test);
	SiMustWriteLock(sb);
	AuDebugOn(!au_flag_test(sb, AuFlag_XINO));
	br = stobr(sb, bindex);
	AuDebugOn(br->br_xino);

	do_create = 1;
	bshared = -1;
	shared_br = NULL;
	bend = sbend(sb);
	if (do_test)
		bshared = is_sb_shared(sb, bindex, bend);
	if (unlikely(bshared >= 0)) {
		shared_br = stobr(sb, bshared);
		do_create = !shared_br->br_xino;
	}

	if (do_create) {
		parent = dget_parent(base_file->f_dentry);
		dir = parent->d_inode;

		vfsub_i_lock_nested(dir, AuLsc_I_PARENT);
		file = xino_create2(sb, base_file, NULL);
		err = PTR_ERR(file);
		vfsub_i_unlock(dir);
		dput(parent);
		if (IS_ERR(file))
			goto out;
		br->br_xino = file;
	} else {
		br->br_xino = shared_br->br_xino;
		get_file(br->br_xino);
	}

	inode = sb->s_root->d_inode;
	h_inode = au_h_iptr_i(inode, bindex);
	xino.ino = inode->i_ino;
	//xino.h_gen = h_inode->i_generation;
	//WARN_ON(xino.h_gen == AuXino_INVALID_HGEN);
	err = xino_write(sb, bindex, h_inode->i_ino, &xino);
	//if (LktrCond) err = -1;
	if (!err)
		return 0; /* success */

	//fput(br->br_xino);
	//br->br_xino = NULL;

 out:
	TraceErr(err);
	return err;
}

/* too slow */
static int do_xib_restore(struct super_block *sb, struct file *file, void *page)
{
	int err, bit;
	struct aufs_sbinfo *sbinfo;
	readf_t func;
	loff_t pos, pend;
	ssize_t sz;
	struct xino *xino;
	unsigned long pindex;

	TraceEnter();
	SiMustWriteLock(sb);

	err = 0;
	sbinfo = stosi(sb);
	func = sbinfo->si_xread;
	err = au_update_fuse_h_inode(file->f_vfsmnt, file->f_dentry, /*dlgt*/0);
	AuDebugOn(err);
	pend = i_size_read(file->f_dentry->d_inode);
#ifdef CONFIG_AUFS_DEBUG
	if (pend > (1 << 22))
		Warn("testing a large xino file %Ld\n", (long long)pend);
#endif
	pos = 0;
	while (pos < pend) {
		err = sz = xino_fread(func, file, page, PAGE_SIZE, &pos);
		if (unlikely(sz <= 0))
			goto out;

		err = 0;
		for (xino = page; sz > 0; xino++, sz -= sizeof(xino)) {
			//Dbg("i%lu\n", xino->ino);
			if (unlikely(xino->ino < AUFS_FIRST_INO))
				continue;

			xib_calc_bit(xino->ino, &pindex, &bit);
			AuDebugOn(page_bits <= bit);
			err = xib_pindex(sb, pindex);
			if (!err)
				set_bit(bit, sbinfo->si_xib_buf);
			else
				goto out;
			//Dbg("i%lu, bit %d\n", xino->ino, bit);
		}
	}

 out:
	TraceErr(err);
	return err;
}

static int xib_restore(struct super_block *sb)
{
	int err;
	aufs_bindex_t bindex, bend;
	void *page;

	//au_debug_on();
	TraceEnter();

	err = -ENOMEM;
	page = (void*)__get_free_page(GFP_KERNEL);
	if (unlikely(!page))
		goto out;

	err = 0;
	bend = sbend(sb);
	for (bindex = 0; !err && bindex <= bend; bindex++)
		if (!bindex || is_sb_shared(sb, bindex, bindex - 1) < 0)
			err = do_xib_restore(sb, stobr(sb, bindex)->br_xino, page);
		else
			LKTRTrace("b%d\n", bindex);
	free_page((unsigned long)page);

 out:
	TraceErr(err);
	//au_debug_off();
	return err;
}

int xib_trunc(struct super_block *sb)
{
	int err;
	struct aufs_sbinfo *sbinfo;
	unsigned long *p;
	//ino_t ino;
	loff_t pos;
	ssize_t sz;
	struct dentry *parent;
	struct inode *dir;
	struct file *file;

	TraceEnter();
	SiMustWriteLock(sb);

	if (unlikely(!au_flag_test(sb, AuFlag_XINO)))
		return 0;

	//aufs_debug_on();
	sbinfo = stosi(sb);
	parent = dget_parent(sbinfo->si_xib->f_dentry);
	dir = parent->d_inode;
	vfsub_i_lock_nested(dir, AuLsc_I_PARENT);
	file = xino_create2(sb, sbinfo->si_xib, NULL);
	vfsub_i_unlock(dir);
	dput(parent);
	err = PTR_ERR(file);
	if (IS_ERR(file))
		goto out;
	fput(sbinfo->si_xib);
	sbinfo->si_xib = file;

	p = sbinfo->si_xib_buf;
	memset(p, 0, PAGE_SIZE);
	pos = 0;
	sz = xino_fwrite(sbinfo->si_xwrite, sbinfo->si_xib, p, PAGE_SIZE,
			 &pos);
	if (unlikely(sz != PAGE_SIZE)) {
		err = sz;
		IOErr("err %d\n", err);
		if (sz >= 0)
			err = -EIO;
		goto out;
	}

	if (au_flag_test(sb, AuFlag_XINO)) {
		mutex_lock(&sbinfo->si_xib_mtx);
		err = xib_restore(sb);
		mutex_unlock(&sbinfo->si_xib_mtx);
#if 0
	} else {
		/* is it really safe? */
		AuDebugOn(!kernel_locked());
		ino = AUFS_FIRST_INO;
		list_for_each_entry(inode, &sb->s_inodes, i_sb_list)
			if (ino < inode->i_ino)
				ino = inode->i_ino;

		/* make iunique to return larger than active max inode number */
		iunique(sb, ino);
		err = 0;
#endif
	}

out:
	//aufs_debug_off();
	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

/*
 * xino mount option handlers
 */
static readf_t find_readf(struct file *h_file)
{
	const struct file_operations *fop = h_file->f_op;

	if (fop) {
		if (fop->read)
			return fop->read;
		if (fop->aio_read)
			return do_sync_read;
	}
	return ERR_PTR(-ENOSYS);
}

static writef_t find_writef(struct file *h_file)
{
	const struct file_operations *fop = h_file->f_op;

	if (fop) {
		if (fop->write)
			return fop->write;
		if (fop->aio_write)
			return do_sync_write;
	}
	return ERR_PTR(-ENOSYS);
}

/* xino bitmap */
static void xino_clear_xib(struct super_block *sb)
{
	struct aufs_sbinfo *sbinfo;

	TraceEnter();
	SiMustWriteLock(sb);

	sbinfo = stosi(sb);
	sbinfo->si_xread = NULL;
	sbinfo->si_xwrite = NULL;
	if (unlikely(sbinfo->si_xib))
		fput(sbinfo->si_xib);
	sbinfo->si_xib = NULL;
	free_page((unsigned long)sbinfo->si_xib_buf);
	sbinfo->si_xib_buf = NULL;
}

static int xino_set_xib(struct super_block *sb, struct file *base)
{
	int err;
	struct aufs_sbinfo *sbinfo;
	struct file *file;
	loff_t pos;

	LKTRTrace("%.*s\n", DLNPair(base->f_dentry));
	SiMustWriteLock(sb);

	sbinfo = stosi(sb);
	file = xino_create2(sb, base, sbinfo->si_xib);
	err = PTR_ERR(file);
	if (IS_ERR(file))
		goto out;
	if (unlikely(sbinfo->si_xib))
		fput(sbinfo->si_xib);
	sbinfo->si_xib = file;
	sbinfo->si_xread = find_readf(file);
	AuDebugOn(IS_ERR(sbinfo->si_xread));
	sbinfo->si_xwrite = find_writef(file);
	AuDebugOn(IS_ERR(sbinfo->si_xwrite));

	err = -ENOMEM;
	if (!sbinfo->si_xib_buf)
		sbinfo->si_xib_buf = (void*)get_zeroed_page(GFP_KERNEL);
	if (unlikely(!sbinfo->si_xib_buf))
		goto out_unset;

	sbinfo->si_xib_last_pindex = 0;
	sbinfo->si_xib_next_bit = 0;

	/* no need to lock for i_size_read() */
	err = au_update_fuse_h_inode(file->f_vfsmnt, file->f_dentry, /*dlgt*/0);
	AuDebugOn(err);
	if (i_size_read(file->f_dentry->d_inode) < PAGE_SIZE) {
		pos = 0;
		err = xino_fwrite(sbinfo->si_xwrite, file, sbinfo->si_xib_buf,
				  PAGE_SIZE, &pos);
		if (unlikely(err != PAGE_SIZE))
			goto out_free;
	}
	err = 0;
	goto out; /* success */

 out_free:
	free_page((unsigned long)sbinfo->si_xib_buf);
	sbinfo->si_xib_buf = NULL;
	if (err >= 0)
		err = -EIO;
 out_unset:
	fput(sbinfo->si_xib);
	sbinfo->si_xib = NULL;
	sbinfo->si_xread = NULL;
	sbinfo->si_xwrite = NULL;
 out:
	TraceErr(err);
	return err;
}

/* xino for each branch */
static void xino_clear_br(struct super_block *sb)
{
	aufs_bindex_t bindex, bend;
	struct aufs_branch *br;

	TraceEnter();
	SiMustWriteLock(sb);

	bend = sbend(sb);
	for (bindex = 0; bindex <= bend; bindex++) {
		br = stobr(sb, bindex);
		if (unlikely(!br || !br->br_xino))
			continue;

		fput(br->br_xino);
		br->br_xino = NULL;
	}
}

static int xino_set_br(struct super_block *sb, struct file *base)
{
	int err;
	aufs_bindex_t bindex, bend, bshared;
	struct aufs_branch *br, *shared;
	struct xino xino;
	struct file *file;
	struct inode *inode;

	LKTRTrace("%.*s\n", DLNPair(base->f_dentry));
	SiMustWriteLock(sb);

	bend = sbend(sb);
	for (bindex = bend; bindex >= 0; bindex--) {
		br = stobr(sb, bindex);
		if (!br->br_xino || file_count(br->br_xino) == 1)
			continue;

		fput(br->br_xino);
		br->br_xino = NULL;
	}

	for (bindex = 0; bindex <= bend; bindex++) {
		br = stobr(sb, bindex);
		if (!br->br_xino)
			continue;
		AuDebugOn(file_count(br->br_xino) != 1);
		file = xino_create2(sb, base, br->br_xino);
		if (IS_ERR(file)) {
			//todo: restore
			err = PTR_ERR(file);
			goto out;
		}
		fput(br->br_xino);
		br->br_xino = file;
	}

	err = 0;
	inode = sb->s_root->d_inode;
	xino.ino = AUFS_ROOT_INO;
	for (bindex = 0; !err && bindex <= bend; bindex++) {
		br = stobr(sb, bindex);
		if (br->br_xino)
			continue;
		bshared = is_sb_shared(sb, bindex, bindex);
		if (bshared < 0) {
			file = xino_create2(sb, base, NULL);
			if (IS_ERR(file)) {
				//todo: restore
				err = PTR_ERR(file);
				goto out;
			}
			br->br_xino = file;
		} else {
			shared = stobr(sb, bshared);
			AuDebugOn(!shared->br_xino);
			br->br_xino = shared->br_xino;
			get_file(br->br_xino);
		}
		err = xino_write(sb, bindex, au_h_iptr_i(inode, bindex)->i_ino,
				 &xino);
	}

 out:
	TraceErr(err);
	return err;
}

void xino_clr(struct super_block *sb)
{
	TraceEnter();
	SiMustWriteLock(sb);

	xino_clear_xib(sb);
	xino_clear_br(sb);
	au_flag_clr(sb, AuFlag_XINO);
}

int xino_set(struct super_block *sb, struct opt_xino *xino, int remount)
{
	int err, skip;
	struct dentry *parent, *cur_parent;
	struct qstr *dname, *cur_name;
	struct file *cur_xino;
	struct inode *dir;

	LKTRTrace("%s\n", xino->path);
	SiMustWriteLock(sb);

	err = 0;
	parent = dget_parent(xino->file->f_dentry);
	if (remount) {
		skip = 0;
		dname = &xino->file->f_dentry->d_name;
		cur_xino = stosi(sb)->si_xib;
		if (cur_xino) {
			cur_parent = dget_parent(cur_xino->f_dentry);
			cur_name = &cur_xino->f_dentry->d_name;
			skip = (cur_parent == parent
				&& dname->len == cur_name->len
				&& !memcmp(dname->name, cur_name->name,
					   dname->len));
			dput(cur_parent);
		}
		if (skip)
			goto out;
	}

	au_flag_set(sb, AuFlag_XINO);
	dir = parent->d_inode;
	vfsub_i_lock_nested(dir, AuLsc_I_PARENT);
	err = xino_set_xib(sb, xino->file);
	if (!err)
		err = xino_set_br(sb, xino->file);
	vfsub_i_unlock(dir);
	if (!err)
		goto out; /* success */

	/* reset all */
	IOErr("failed creating xino, forcing noxino (%d).\n", err);
	err = -EIO;
	xino_clr(sb);

 out:
	dput(parent);
	TraceErr(err);
	return err;
}

/* ---------------------------------------------------------------------- */

/*
 * create a xinofile at the default place/path.
 */
struct file *xino_def(struct super_block *sb)
{
	struct file *file;
	aufs_bindex_t bend, bindex, bwr;
	char *page, *p;

	TraceEnter();

	bend = sbend(sb);
	bwr = -1;
	for (bindex = 0; bindex <= bend; bindex++)
		if (br_writable(sbr_perm(sb, bindex))
		    && !au_is_nfs(au_h_dptr_i(sb->s_root, bindex)->d_sb)) {
			bwr = bindex;
			break;
		}

	if (bwr >= 0) {
		// todo: rewrite with lkup_one()
		file = ERR_PTR(-ENOMEM);
		page = __getname();
		//if (LktrCond) {__putname(page); page = NULL;}
		if (unlikely(!page))
			goto out;
		p = d_path(au_h_dptr_i(sb->s_root, bwr), sbr_mnt(sb, bwr), page,
			   PATH_MAX - sizeof(AUFS_XINO_FNAME));
		//if (LktrCond) p = ERR_PTR(-1);
		file = (void*)p;
		if (p && !IS_ERR(p)) {
			strcat(p, "/" AUFS_XINO_FNAME);
			LKTRTrace("%s\n", p);
			file = xino_create(sb, p, /*silent*/0, sb->s_root);
			//if (LktrCond) {fput(file); file = ERR_PTR(-1);}
		}
		__putname(page);
	} else {
		file = xino_create(sb, AUFS_XINO_DEFPATH, /*silent*/0,
				   /*parent*/NULL);
		//if (LktrCond) {fput(file); file = ERR_PTR(-1);}
	}

 out:
	TraceErrPtr(file);
	return file;
}
