// distribution boxbackup-0.11rc2 (svn version: 2072)
//  
// Copyright (c) 2003 - 2008
//      Ben Summers and contributors.  All rights reserved.
//  
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. All use of this software and associated advertising materials must 
//    display the following acknowledgment:
//        This product includes software developed by Ben Summers.
// 4. The names of the Authors may not be used to endorse or promote
//    products derived from this software without specific prior written
//    permission.
// 
// [Where legally impermissible the Authors do not disclaim liability for 
// direct physical injury or death caused solely by defects in the software 
// unless it is modified by a third party.]
// 
// THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
// ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//  
//  
//  
// --------------------------------------------------------------------------
//
// File
//		Name:    ReadLoggingStream.cpp
//		Purpose: Buffering wrapper around IOStreams
//		Created: 2007/01/16
//
// --------------------------------------------------------------------------

#include "Box.h"

#include <string.h>

#include "ReadLoggingStream.h"
#include "CommonException.h"
#include "Logging.h"

#include "MemLeakFindOn.h"

// --------------------------------------------------------------------------
//
// Function
//		Name:    ReadLoggingStream::ReadLoggingStream(const char *, int, int)
//		Purpose: Constructor, set up buffer
//		Created: 2007/01/16
//
// --------------------------------------------------------------------------
ReadLoggingStream::ReadLoggingStream(IOStream& rSource)
: mrSource(rSource),
  mOffset(0),
  mLength(mrSource.BytesLeftToRead()),
  mTotalRead(0),
  mStartTime(GetCurrentBoxTime())
{ }


// --------------------------------------------------------------------------
//
// Function
//		Name:    ReadLoggingStream::Read(void *, int)
//		Purpose: Reads bytes from the file
//		Created: 2007/01/16
//
// --------------------------------------------------------------------------
int ReadLoggingStream::Read(void *pBuffer, int NBytes, int Timeout)
{
	int numBytesRead = mrSource.Read(pBuffer, NBytes, Timeout);

	if (numBytesRead > 0)
	{
		mTotalRead += numBytesRead;
		mOffset += numBytesRead;
	}

	if (mLength >= 0 && mTotalRead > 0)
	{	
		box_time_t timeNow = GetCurrentBoxTime();
		box_time_t elapsed = timeNow - mStartTime;
		box_time_t finish  = (elapsed * mLength) / mTotalRead;
		box_time_t remain  = finish - elapsed;

		BOX_TRACE("Read " << numBytesRead << " bytes at " << mOffset << 
			", " << (mLength - mOffset) << " remain, eta " <<
			BoxTimeToSeconds(remain) << "s");
	}
	else if (mLength >= 0 && mTotalRead == 0)
	{
		BOX_TRACE("Read " << numBytesRead << " bytes at " << mOffset << 
			", " << (mLength - mOffset) << " remain");
	}
	else
	{	
		BOX_TRACE("Read " << numBytesRead << " bytes at " << mOffset << 
			", unknown bytes remaining");
	}
	
	return numBytesRead;
}


// --------------------------------------------------------------------------
//
// Function
//		Name:    ReadLoggingStream::BytesLeftToRead()
//		Purpose: Returns number of bytes to read (may not be most efficient function ever)
//		Created: 2007/01/16
//
// --------------------------------------------------------------------------
IOStream::pos_type ReadLoggingStream::BytesLeftToRead()
{
	return mLength - mOffset;
}


// --------------------------------------------------------------------------
//
// Function
//		Name:    ReadLoggingStream::Write(void *, int)
//		Purpose: Writes bytes to the underlying stream (not supported)
//		Created: 2003/07/31
//
// --------------------------------------------------------------------------
void ReadLoggingStream::Write(const void *pBuffer, int NBytes)
{
	THROW_EXCEPTION(CommonException, NotSupported);
}


// --------------------------------------------------------------------------
//
// Function
//		Name:    ReadLoggingStream::GetPosition()
//		Purpose: Get position in stream
//		Created: 2003/08/21
//
// --------------------------------------------------------------------------
IOStream::pos_type ReadLoggingStream::GetPosition() const
{
	return mOffset;
}


// --------------------------------------------------------------------------
//
// Function
//		Name:    ReadLoggingStream::Seek(pos_type, int)
//		Purpose: Seeks within file, as lseek, invalidate buffer
//		Created: 2003/07/31
//
// --------------------------------------------------------------------------
void ReadLoggingStream::Seek(IOStream::pos_type Offset, int SeekType)
{
	mrSource.Seek(Offset, SeekType);

	switch (SeekType)
	{
		case SeekType_Absolute:
		{
			// just go there
			mOffset = Offset;
		}
		break;

		case SeekType_Relative:
		{
			// Actual underlying file position is 
			// (mBufferSize - mBufferPosition) ahead of us.
			// Need to subtract that amount from the seek
			// to seek forward that much less, putting the 
			// real pointer in the right place.
			mOffset += Offset;
		}
		break;

		case SeekType_End:
		{
			// Actual underlying file position is 
			// (mBufferSize - mBufferPosition) ahead of us.
			// Need to add that amount to the seek
			// to seek backwards that much more, putting the 
			// real pointer in the right place.
			mOffset = mLength - Offset;
		}
	}
}


// --------------------------------------------------------------------------
//
// Function
//		Name:    ReadLoggingStream::Close()
//		Purpose: Closes the underlying stream (not needed)
//		Created: 2003/07/31
//
// --------------------------------------------------------------------------
void ReadLoggingStream::Close()
{
	THROW_EXCEPTION(CommonException, NotSupported);
}


// --------------------------------------------------------------------------
//
// Function
//		Name:    ReadLoggingStream::StreamDataLeft()
//		Purpose: Any data left to write?
//		Created: 2003/08/02
//
// --------------------------------------------------------------------------
bool ReadLoggingStream::StreamDataLeft()
{
	return mrSource.StreamDataLeft();
}

// --------------------------------------------------------------------------
//
// Function
//		Name:    ReadLoggingStream::StreamClosed()
//		Purpose: Is the stream closed?
//		Created: 2003/08/02
//
// --------------------------------------------------------------------------
bool ReadLoggingStream::StreamClosed()
{
	return mrSource.StreamClosed();
}

