/*******************************************************\
* irmp3-ncurses - An ncurses frontend for irmp3 using   *
* the Network Control Module                            *
* (C) 2003 Ross Axe                                     *
*                                                       *
* display.c - Handles drawing of UI                     *
\*******************************************************/

#if HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdarg.h>

#include "irmp3-ncurses.h"

vcid("$Id: display.c,v 1.53 2005/03/28 12:46:06 ross Exp $");

WINDOW *statewin;
WINDOW *debugwin;

/* Position at which sliders were last drawn */
int timeleft = -1;
int repeatleft = -1;
int volumeleft = -1;
int balanceleft = -1;
int bassleft = -1;
int trebleleft = -1;

/*
 * vlogprintf
 * prints to logfile
 * Usage identical to printf (hopefully)
 * Call with a NULL fmt to close log
 */
#ifdef LOGGING
static int vlogprintf(const char *fmt, va_list ap)
{
#  define LOGFILENAME "irmp3-ncurses.log"
    static FILE *logfile = NULL;

    if(!fmt) {
	if(logfile)
	    return fclose(logfile);
	else
	    return 0;
    }
    if(loglevel > 0) {
	if(!logfile) {
	    logfile = fopen(LOGFILENAME, "w");
	    if(logfile)
		sbar_printf("Logging to file %s at level %d\n", LOGFILENAME,
			    loglevel);
	}
    }
    if(loglevel > 0 && logfile)
	return vfprintf(logfile, fmt, ap);
    else
	return 0;
}

static int logprintf(const char *fmt, ...)
{
    va_list ap;
    int ret;

    va_start(ap, fmt);
    ret = vlogprintf(fmt, ap);
    va_end(ap);
    return ret;
}

#else
#  define vlogprintf(fmt, ap) 0
#  define logprintf(...) 0
#endif

static int vsbar_printf_(bool scroll, const char *fmt, va_list ap)
{
    if(!fmt)
	return vlogprintf(fmt, ap);
    else {
	va_list aq;
	int ret;

	va_copy(aq, ap);
	vlogprintf(fmt, aq);
	va_end(aq);
	logprintf("\n");
 
	wclrtoeol(debugwin);
	ret = vw_printw(debugwin, fmt, ap);
	if(scroll)
	    waddch(debugwin, '\n');
	wcr(debugwin);
	wrefresh(debugwin);
	return ret;
    }
}

int sbar_printf_noscroll(const char *fmt, ...)
{
    va_list ap;
    int ret;

    va_start(ap, fmt);
    ret = vsbar_printf_(false, fmt, ap);
    va_end(ap);
    return ret;
}

int sbar_printf(const char *fmt, ...)
{
    va_list ap;
    int ret;

    va_start(ap, fmt);
    ret = vsbar_printf_(true, fmt, ap);
    va_end(ap);
    return ret;
}

int sbar_getstr(char *buf, size_t size, const char *fmt, ...)
{
    va_list ap;
    int ret;

    va_start(ap, fmt);
    vw_printw(debugwin, fmt, ap);
    va_end(ap);
    curs_set(1);
    echo();
    wrefresh(debugwin);
    ret = wgetnstr(debugwin, buf, size);
    noecho();
    curs_set(0);
    waddch(debugwin, '\r');
    return ret;
}

int wcr(WINDOW *win)
{
    return wmove(win, getcury(win), 0);
}

void printtitlebar(WINDOW *win, const char *caption, const char *fmt, ...)
{
    int i;

    wattrset(win, TITLE_ATTR);
    mvwaddstr(win, 0, 0, "  ");
    wattrset(win, A_BOLD | TITLE_ATTR);
    waddstr(win, caption);
    wattrset(win, TITLE_ATTR);
    if(fmt) {
	va_list ap;

	va_start(ap, fmt);
	vw_printw(win, fmt, ap);
	va_end(ap);
    }
    for(i = getcurx(win); i < getmaxx(win); i++)
	waddch(win, ' ');
    wstandend(win);
}

void printtime(int secs, int rsecs)
{
    songinfo.pos = secs;
    if(rsecs >= 0)
	songinfo.length = secs + rsecs;
    else
	songinfo.length = -1;
    if(rsecs >= 0)
	mvwprintw(statewin, TIME_ROW, 0, _("Time: %d:%02d (-%d:%02d) "),
		  secs / 60, secs % 60, rsecs / 60, rsecs % 60);
    else
	mvwprintw(statewin, TIME_ROW, 0, _("Time: %d:%02d "),
		  secs / 60, secs % 60);
    timeleft = getcurx(statewin);
    wprintslider(statewin, getmaxx(statewin) - timeleft, secs, 0,
		 secs + rsecs);
    wrefresh(statewin);
}

#if USE_MOUSE
static void print_playcontrols(void)
{
    const chtype buttons[][3] = {
	{ACS_VLINE, ACS_LARROW, '\0'},	/* skip previous */
	{ACS_LARROW, ACS_LARROW, '\0'},	/* rewind */
	{ACS_VLINE, ACS_RARROW, '\0'},	/* play */
	{ACS_VLINE, ACS_VLINE, '\0'},	/* pause */
	{ACS_BLOCK, ACS_BLOCK, '\0'},	/* stop */
	{ACS_RARROW, ACS_RARROW, '\0'},	/* fast forward */
	{ACS_RARROW, ACS_VLINE, '\0'}	/* skip next */
    };
    int i;

    for(i = 0; i < (NUMELEMENTS(buttons)); i++) {
	mvwaddchstr(statewin, BUTTON_ROW, i * 3 + BUTTON_LEFT, buttons[i]);
    }
}
#endif

void printplayerstate(void)
{
    static const char *repeatnames[] = {
	[RP_NONE] = N_("none"),
	[RP_TRACK] = N_("track"),
	[RP_ALL] = N_("playlist"),
    };
    char *fname;
    const int w = getmaxx(statewin);
    static struct scroller trackname_scroller = SCROLLER_INITIALISER;

    printtitlebar(statewin, PACKAGE_NAME, " v%s", PACKAGE_VERSION);
#if USE_MOUSE
    print_playcontrols();
#endif
    fname = strrchr(songinfo.filename, '/');
    if(fname)
	fname++;		/* skip over '/' */
    else
	fname = songinfo.filename;

    scroller_end(&trackname_scroller);
    switch(playerstate.playstate) {
    case PS_STOP:
	mvwaddstr(statewin, FILENAME_ROW, 0, _("Stopped"));
	wclrtoeol(statewin);
	break;
    case PS_PLAY:
	mvwprintw(statewin, FILENAME_ROW, 0, _("Playing: "));
	wscroller(&statewin, &trackname_scroller, fname, false);
	break;
    case PS_PAUSE:
	mvwprintw(statewin, FILENAME_ROW, 0, _("Paused:  "));
	wscroller(&statewin, &trackname_scroller, fname, false);
	break;
    case PS_HALT:
	mvwprintw(statewin, FILENAME_ROW, 0, _("Halted"));
	wclrtoeol(statewin);
	break;
    }

    mvwprintw(statewin, SHUFFLE_REPEAT_ROW, 0, _("Shuffle: %s  "),
	      playerstate.shuffle ? _("on ") : _("off"));
    repeatleft = getcurx(statewin);
    wprintw(statewin, _("Repeat: %s"), _(repeatnames[playerstate.repeat]));
    wclrtoeol(statewin);

    mvwprintw(statewin, VOLUME_ROW, 0, _("Volume: %3d%% %s"),
	      playerstate.volume, playerstate.mute ? _("(mute) ") : "");
    volumeleft = getcurx(statewin);
    wprintslider(statewin, w - volumeleft, playerstate.volume, 0, 100);

    mvwprintw(statewin, BASS_ROW, 0, _("Bass:   %3d%% "), playerstate.bass);
    bassleft = getcurx(statewin);
    wprintslider(statewin, w - bassleft, playerstate.bass, 0, 100);

    mvwprintw(statewin, TREBLE_ROW, 0, _("Treble: %3d%% "),
	      playerstate.treble);
    trebleleft = getcurx(statewin);
    wprintslider(statewin, w - trebleleft, playerstate.treble, 0, 100);

    if(playerstate.balance == 0)
	mvwprintw(statewin, BALANCE_ROW, 0, _("Balance: centre "));
    else if(playerstate.balance < 0)
	mvwprintw(statewin, BALANCE_ROW, 0, _("Balance:%3d%% left "),
		  -playerstate.balance * 2);
    else
	mvwprintw(statewin, BALANCE_ROW, 0, _("Balance:%3d%% right "),
		  playerstate.balance * 2);
    balanceleft = getcurx(statewin);
    wprintslider(statewin, w - balanceleft, playerstate.balance, -50, 50);

    wrefresh(statewin);
}

/*
 * printslider
 * Prints a horizontal slider  at current location
 * length:   length in characters
 * value:    from min to max
 */
void mvwprintslider(WINDOW *wnd, int r, int c, int length, int value, int min,
		    int max)
{
    wmove(wnd, r, c);
    wprintslider(wnd, length, value, min, max);
}

void wprintslider(WINDOW *wnd, int length, int value, int min, int max)
{
    int i;

    if(length < 2)
	return;
    length -= 2;		/* take into account end bars */
    waddch(wnd, ACS_LARROW);
    for(i = 0; i < length; i++) {
	waddch(wnd, ((max > min && ((value - min) * length
				    / (max - min + 1)) == i)
		     ? ACS_DIAMOND : ACS_HLINE));
    }
    waddch(wnd, ACS_RARROW);
}

void mvwprintvslider(WINDOW *wnd, int r, int c, int length, int value, int min,
		     int max)
{
    int i;

    length -= 2;		/* take into account end bars */
    mvwaddch(wnd, r, c, ACS_UARROW);
    for(i = 0; i < length; i++) {
	mvwaddch(wnd, r + i + 1, c, max > min && ((value - min) * length
						  / (max - min + 1) == i)
		 ? ACS_BLOCK : ACS_VLINE);
    }
    mvwaddch(wnd, r + length + 1, c, ACS_DARROW);
}

void printsonginfo(void)
{
    static struct scroller artist_scroller = SCROLLER_INITIALISER;
    static struct scroller album_scroller = SCROLLER_INITIALISER;
    static struct scroller title_scroller = SCROLLER_INITIALISER;
    static struct scroller genre_scroller = SCROLLER_INITIALISER;
    static struct scroller comment_scroller = SCROLLER_INITIALISER;

    scroller_end(&artist_scroller);
    mvwprintw(statewin, ARTIST_ROW, 0, _("Artist:  "));
    wscroller(&statewin, &artist_scroller, songinfo.artist, false);
    scroller_end(&album_scroller);
    mvwprintw(statewin, ALBUM_ROW, 0, _("Album:   "));
    wscroller(&statewin, &album_scroller, songinfo.album, false);
    scroller_end(&title_scroller);
    mvwprintw(statewin, TITLE_ROW, 0, _("Title:   "));
    wscroller(&statewin, &title_scroller, songinfo.title, false);
    scroller_end(&genre_scroller);
    mvwprintw(statewin, YEAR_GENRE_ROW, 0, _("Year:    %4s  Genre: "),
	      songinfo.year);
    wscroller(&statewin, &genre_scroller, songinfo.genre, false);
    scroller_end(&comment_scroller);
    mvwprintw(statewin, COMMENT_ROW, 0, _("Comment: "));
    wscroller(&statewin, &comment_scroller, songinfo.comment, false);
    wrefresh(statewin);
}


void about(void)
{
    sbar_printf(_("client version: %s"), PACKAGE_STRING);
    if(playerstate.version[0])
	sbar_printf(_("server version: %s"), playerstate.version);
    else
	sbar_printf(_("server version is unknown"));
    if(playerstate.playerversion[0]) {
	sbar_printf(_("%s version: %s"), (playerstate.playermodule[0]
					  ? playerstate.playermodule
					  : _("Unknown player")),
		    playerstate.playerversion);
    } else {
	sbar_printf(_("%s version is unknown"), (playerstate.playermodule[0]
						 ? playerstate.playermodule
						 : _("Unknown player")));
    }
}


void redraw_all(void)
{
    int h, w;

    getmaxyx(stdscr, h, w);

    /* Redraw root window */
    clearok(stdscr, TRUE);
    refresh();

    /* Repaint debug window */
    clearok(debugwin, TRUE);
    wrefresh(debugwin);

    /* Redraw Playlist */
    if(show_playlist) {
	wclear(plwin);
	playlist_print();
    }

    /* Redraw status window */
    wclear(statewin);
    printplayerstate();
    printsonginfo();
}


void init_ui(void)
{
    int h, w;

    getmaxyx(stdscr, h, w);
    debugwin = newwin(DEBUG_HEIGHT, w, h - DEBUG_HEIGHT, 0);
    scrollok(debugwin, true);
    if(show_playlist) {
	statewin = newwin(STATE_HEIGHT, PLAYLIST_LEFT - 1, STATE_TOP, 0);
	plwin_create(-1);
    } else
	statewin = newwin(STATE_HEIGHT, w, STATE_TOP, 0);
    redraw_all();
}


void destroy_ui(void)
{
    scroller_endall();
}

#if TERM_RESIZE
void resize_terminal(void)
{
    int h, w;

    getmaxyx(stdscr, h, w);
    mvwin(debugwin, h - DEBUG_HEIGHT, 0);
    if(show_playlist) {
	plouterwin = resize_window(plouterwin, h - PLAYLIST_TOP - DEBUG_HEIGHT,
				   w - PLAYLIST_LEFT);
	plwin = resize_window(plwin, h - PLAYLIST_TOP - DEBUG_HEIGHT - 1,
			      w - PLAYLIST_LEFT - 1);
    }
    redraw_all();
}
#endif
