/*******************************************************\
* irmp3-ncurses - An ncurses frontend for irmp3 using   *
* the Network Control Module                            *
* (C) 2003 Ross Axe                                     *
*                                                       *
* playlist.c - playlist handling                        *
\*******************************************************/

#if HAVE_CONFIG_H
#  include "config.h"
#endif

#include "irmp3-ncurses.h"

vcid("$Id: playlist.c,v 1.42 2005/03/26 19:33:41 ross Exp $");

bool show_playlist = true;
WINDOW *plwin, *plouterwin;
static WINDOW *divwin;
#define PLCURS_MARGIN 2 /* Distance kept between cursor and edge of screen */

static void playlist_printitem(int i, int pltop, int plcurs, int plplaying);
static int playlist_cursmove(int offset);

struct plentry {
    char *name;
    /* maybe add some more gen here later - like tags or stuff */
};

struct plentry *playlist = NULL;
int plcount = 0;
bool playlist_clean = false;


static int pltop = 0;		/* index of topmost visible element */
static int plcurs = 0;		/* index of playlist cursor */
static int plplaying = -1;	/* index of currently playing song */

static struct scroller curs_scroller = SCROLLER_INITIALISER;

static bool reading_playlist = false;	/* true if in the middle of reading,
					   i.e. false normally
					   set to true when 1st item recieved
					   and back to false at end-of-list
					   marker */



static void playlist_printscrollbar(void)
{
    int h = getmaxy(plwin), w = getmaxx(plouterwin);

    assert(show_playlist);
    mvwprintvslider(plouterwin, 1, w - 1, h, pltop, 0, plcount - h);
}

static void playlist_printtitlebar(void)
{
    printtitlebar(plouterwin, _("Playlist"), " - %d/%d", plcurs + 1, plcount);
}

static void playlist_printdivider(void)
{
    const int h = getmaxy(divwin);

    mvwaddch(divwin, 0, 0, ACS_TTEE);
    mvwvline(divwin, 1, 0, 0, h - 2);
    mvwaddch(divwin, h - 1, 0, ACS_LLCORNER);
    wrefresh(divwin);
}

static void playlist_replace(int index, const char *name)
{
    free(playlist[index].name);
    playlist[index].name = strdup(name);

    if(strcmp(name, songinfo.filename) == 0) {
	playlist_setplaying(index);
    } else if(show_playlist) {
	if(index >= pltop && index < pltop + getmaxy(plwin)) {
	    playlist_printitem(index - pltop, pltop, plcurs, plplaying);
	    wrefresh(plwin);
	}
	if(index > plcurs && index <= plcurs + PLCURS_MARGIN)
	    playlist_cursmove(+0);
    }
}

static void playlist_add(const char *name)
{
    int index = plcount++;

    playlist = realloc(playlist, plcount * sizeof playlist[0]);
    if(!playlist) {
	sbar_printf(_("Out of memory allocating playlist!"));
	plcount = 0;
	playlist_print();
	return;
    }
    playlist[index] = (struct plentry) {
	.name = NULL,
    };
    playlist_replace(index, name);
}

static void playlist_save(const char *filename)
{
    int i;
    FILE *file;

    assert(show_playlist);

    file = fopen(filename, "w");
    if(!file) {
	sbar_printf(_("Couldn't open %s!"), filename);
	return;
    }
    for(i = 0; i < plcount; i++)
	fprintf(file, "%s\n", playlist[i].name);
    fclose(file);
}

static void playlist_truncate(int newcount)
{
    int i;

    assert(newcount <= plcount);

    if(plplaying >= newcount) {
	scroller_end(&curs_scroller);
	plplaying = -1;
    }
    if(plcurs >= newcount)
	plcurs = newcount - 1;
    for(i = newcount; i < plcount; i++) {
	free(playlist[i].name);
    }
    playlist = realloc(playlist, (plcount = newcount) * sizeof playlist[0]);
}

void plwin_create(int sock)
{
    int h, w;

    getmaxyx(stdscr, h, w);
    h -= PLAYLIST_TOP + DEBUG_HEIGHT;
    w -= PLAYLIST_LEFT;
    plouterwin = newwin(h, w, PLAYLIST_TOP, PLAYLIST_LEFT);
    plwin = derwin(plouterwin, h - 1, w - 1, 1, 0);
    divwin = newwin(h, 1, PLAYLIST_TOP, PLAYLIST_LEFT - 1);
    wattrset(divwin, BORDER_ATTR);
    playlist_setplaying(playlist_find(songinfo.filename));
    if(sock >= 0 && !playlist_clean)
	send_query(sock, "plfiles");
}

void plwin_destroy(void)
{
    scroller_end(&curs_scroller);
    wclear(plwin);
    wrefresh(plwin);
    delwin(plwin);
    plwin = NULL;
    wclear(plouterwin);
    wrefresh(plouterwin);
    delwin(plouterwin);
    plouterwin = NULL;
    wclear(divwin);
    wrefresh(divwin);
    delwin(divwin);
    divwin = NULL;
}

void playlist_dirty(int sock)
{
    playlist_clean = false;
    if(show_playlist && !reading_playlist)
	send_query(sock, "plfiles");
}

bool handle_plfiles(int sock, char *buf)
{
    static int baton_i;
    /* baton is an arbitrary sized array of chars, chtypes or whatever
       the idea being that they are displayed in sequence while reading 
       to make a twirling baton, or whatever */
    const char baton[] = {'|', '/', '-', '\\'};
    /* const chtype baton[] = {ACS_S1, ACS_S3, ACS_S7, ACS_S9}; */
    static int i;

    if(!reading_playlist) {
	baton_i = 0;
	i = 0;
    }
    if(strstart("total ", buf) && isdigit(buf[6])) {
	/* obvious protocol problem here
	   if a filename matches "^total [:digit:]" */
	int rep_count = atoi(buf + 6);	/* reported count */

	playlist_truncate(i);
	assert(i == plcount);
	reading_playlist = false;
	if(rep_count == plcount) {
	    sbar_printf(ngettext("%d track in playlist",
				 "%d tracks in playlist", plcount), plcount);
	    playlist_clean = true;
	} else {
	    sbar_printf(ngettext("Playlist count mismatch. %d track reported, "
				 "but I counted %d!", "Playlist count "
				 "mismatch. %d tracks reported, but I counted "
				 "%d!", rep_count), rep_count, plcount);
	    sbar_printf_noscroll(_("Press 'r' to reload playlist."));
	}
	playlist_print();
    } else {
	reading_playlist = true;
	assert(i <= plcount);
	if(i < plcount)
	    playlist_replace(i, buf);
	else
	    playlist_add(buf);
	if(i % 10 == 0) {
	    sbar_printf_noscroll(_("Reading playlist... %c"), baton[baton_i]);
	    baton_i = (baton_i + 1) % NUMELEMENTS(baton);
	    if(show_playlist) {
		playlist_printtitlebar();
		wrefresh(plouterwin);
	    }
	}
	i++;
    }
    return true;
}


static void *cursor_entryhook(struct scroller *scroller)
{
    wattrset(plwin, CURSOR_ATTR | (plcurs == plplaying ? A_BOLD : 0));
    return NULL;
}

static void cursor_exithook(struct scroller *scroller, void *data)
{
    /* should not do this unconditionally */
    wstandend(plwin);
}

/* playlist_printitem: Print a row of the playlist
 * 
 * i:         Row number (of the display)
 * pltop:     Index of topmost (i=0) row
 * plcurs:    Index of currently selected item
 * plplaying: Index of currently playing item
 *
 */
static void playlist_printitem(int i, int pltop, int plcurs, int plplaying)
{
    char *ptr;
    int h, w;

    assert(show_playlist);
    assert(i + pltop < plcount && i + pltop >= 0);

    getmaxyx(plwin, h, w);

    ptr = strrchr(playlist[i + pltop].name, '/');
    if(ptr)
	ptr++;
    else
	ptr = playlist[i + pltop].name;
    if(i + pltop == plcurs) {
	scroller_end(&curs_scroller);
	wmove(plwin, i, 0);
	wscroller_hook(&plwin, &curs_scroller, ptr, true,
		       cursor_entryhook, cursor_exithook);
    } else {
	if(curs_scroller.y == i)
	    scroller_end(&curs_scroller);
	if(pltop + i == plplaying)
	    wattrset(plwin, A_BOLD);
	mvwprintw(plwin, i, 0, "%-*.*s", w, w, ptr);
	wstandend(plwin);
    }
}

void playlist_print(void)
{
    int i;
    int h, w;

    assert(show_playlist);
    getmaxyx(plwin, h, w);
    wclear(plwin);
    for(i = 0; i < h; i++) {
	if(i + pltop >= plcount)
	    break;
	playlist_printitem(i, pltop, plcurs, plplaying);
    }
    wrefresh(plwin); 
    playlist_printscrollbar();
    playlist_printtitlebar();
    playlist_printdivider();
    wrefresh(plouterwin);
}

static void playlist_cursvisible(void)
{
    int h = getmaxy(plwin);

    if(pltop > plcurs - PLCURS_MARGIN)
	pltop = plcurs - PLCURS_MARGIN;
    if(pltop < plcurs - (h - 1) + PLCURS_MARGIN)
	pltop = plcurs - (h - 1) + PLCURS_MARGIN;
    if(pltop > plcount - h)
	pltop = plcount - h;
    if(pltop < 0)
	pltop = 0;
}

static int playlist_cursmove(int offset)
{
    int h, w;
    int oldtop = pltop, oldpos = plcurs;

    assert(show_playlist);

    if(plcount <= 0)
	return plcurs = 0;
    getmaxyx(plwin, h, w);

    plcurs += offset;
    if(plcurs < 0)
	plcurs = 0;
    if(plcurs > plcount - 1)
	plcurs = plcount - 1;
    playlist_cursvisible();

    if(abs(oldtop - pltop) < h) {
	if((oldpos - oldtop) < h && (oldpos - oldtop) >= 0)
	    playlist_printitem(oldpos - oldtop, oldtop, plcurs, plplaying);
	if(oldtop != pltop) {
	    int i;

	    scrollok(plwin, TRUE);
	    wscrl(plwin, pltop - oldtop);
	    scrollok(plwin, FALSE);
	    if(pltop > oldtop) {
		for(i = h - (pltop - oldtop); i < h; i++) {
		    if(i + pltop >= plcount)
			break;
		    playlist_printitem(i, pltop, plcurs, plplaying);
		}
	    } else {
		for(i = 0; i < (oldtop - pltop); i++) {
		    if(i + pltop >= plcount)
			break;
		    playlist_printitem(i, pltop, plcurs, plplaying);
		}
	    }
	}
	if((plcurs - pltop) < h)
	    playlist_printitem(plcurs - pltop, pltop, plcurs, plplaying);
	wrefresh(plwin);
	playlist_printscrollbar();
	playlist_printtitlebar();
	wrefresh(plouterwin);
    } else
	playlist_print();
    return plcurs;
}

int playlist_find(const char *name)
{
    int i;
    int h, w;

    assert(show_playlist);

    getmaxyx(plwin, h, w);

    for(i = 0; i < plcount; i++)
	if(strcmp(name, playlist[i].name) == 0)
	    return i;
    return -1;
}

void playlist_setplaying(int i)
{
    if(plplaying != i) {
	plplaying = i;
	if(show_playlist) {
	    plcurs = i;
	    playlist_cursvisible();
	    playlist_print();
	}
    }
}

static void playlist_seek(int sock)
{
    assert(show_playlist);

    netputs("playlist shuffle 0", sock);
    netprintf(sock, "playlist jump %d", plcurs);
    if(playerstate.shuffle)
	netputs("playlist shuffle 1", sock);
}

#if USE_MOUSE
static bool playlist_innermouse(MEVENT *mevent, int sock)
{
    int row = mevent->y - getbegy(plwin);

    if(mevent->bstate & BUTTON1_DOUBLE_CLICKED) {
	if(pltop + row < plcount) {
	    plcurs = pltop + row;
	    playlist_print();
	    playlist_seek(sock);
	}
    } else if(mevent->bstate & BUTTON1_CLICKED) {
	if(pltop + row < plcount) {
	    plcurs = pltop + row;
	    playlist_print();
	}
    } else {
	sbar_printf("Unknown mouse event, devid %d at (%d, %d, %d), "
		    "buttons %#lo", mevent->id, mevent->x, mevent->y,
		    mevent->z, (unsigned long)mevent->bstate);
    }
    return true;
}

bool playlist_mouse(MEVENT *mevent, int sock)
{
    const int h = getmaxy(plwin);
    const int row = mevent->y - getbegy(plwin);;

    assert(show_playlist);
    if(row < 0)
	return true;
    else if(wenclose(plwin, mevent->y, mevent->x))
	return playlist_innermouse(mevent, sock);
    else if(mevent->bstate & BUTTON1_CLICKED) {
	if(plcount > h) {
	    if(row == 0) {
		if(pltop > 0)
		    pltop--;
	    } else if(row == h - 1) {
		if(pltop < plcount - h)
		    pltop++;
	    } else
		pltop = (plcount - h) * (row - 1) / (h - 3);
	    playlist_print();
	}
    }
    return true;
}
#endif	 /* USE_MOUSE */


static void save_playlist_prompt(void)
{
    char buf[1024];

    assert(show_playlist);

    sbar_getstr(buf, sizeof buf, _("Enter filename: "));
    if(buf[0])
	playlist_save(buf);
    else
	sbar_printf(_("Save cancelled"));
}


bool playlist_key_callback(int sock, int chr)
{
    assert(show_playlist);

    switch(chr) {
    case KEY_UP:
    case '8':
	playlist_cursmove(-1);
	break;
    case KEY_DOWN:
    case '2':
	playlist_cursmove(+1);
	break;
    case KEY_PPAGE:
	playlist_cursmove(-getmaxy(plwin));
	break;
    case KEY_NPAGE:
	playlist_cursmove(+getmaxy(plwin));
	break;
#ifdef PADENTER
    case PADENTER:
#endif
    case KEY_ENTER:
    case '\r':
	playlist_seek(sock);
	break;
    case 'r':
	playlist_dirty(sock);
	break;
    case 's':
	save_playlist_prompt();
	break;
    default:
	return false;
    }
    return true;
}
