#ifndef WORD_FIX_H
#define WORD_FIX_H

#include <qobject.h>

class Chat;
class UserGroup;
class PowerKadu;
class QListBox;

/*!
 * This class is responsible for fixing small mistakes (and not only)
 * while sending chat messages, like "<i>nei</i>" to "<i>nie</i>", etc.
 * Generally it replaces one word with another, assigned to first one.
 * \class WordFix
 * \brief Words fixing.
 */
class WordFix : public QObject
{
	Q_OBJECT

	private:
		/*!
		 * \var QMap<QString,QString> wordsList
		 * Contains whole list of words for replacing in form:
		 * wordList["to_replace"] = "correct_word";
		 */
		QMap<QString,QString> wordsList;

	public:
		/*!
		 * \fn WordFix(PowerKadu *pk)
		 * Default constructor. Reads words list or (if it's not defined yet) loads default list.
		 * \param pk This argument is here, becouse 'powerKadu' pointer isn't assigned while calling WordFix().
		 */
		WordFix(PowerKadu *pk);

		/*!
		 * \fn ~WordFix()
		 * Default destructor.
		 */
		~WordFix();

		/*!
		 * \fn void connectToChat(Chat* chat)
		 * Connects handling slot to given chat signal.
		 * \param chat Chat to connect slot to.
		 */
		void connectToChat(Chat* chat);

		/*!
		 * \fn void disconnectFromChat(Chat* chat)
		 * Disconnects handling slot from given chat signal.
		 * \param chat Chat to disconnect slot from.
		 */
		void disconnectFromChat(Chat* chat);

		/*!
		 * \fn void doReplace(QString &text)
		 * Does the whole job - replaces wrong words with the right ones.
		 * \param text To replace. After processing it will contain corrected text.
		 */
		void doReplace(QString &text);

	private slots:
		/*!
		 * \fn void chatCreated(const UserGroup *group)
		 * This slot calls connectToClass"("Chat* chat")" to connect
		 * the object to chat "<i>send message</i>" signal.
		 * \param group Users in chat window, which emited the signal.
		 */
		void chatCreated(const UserGroup *group);

		/*!
		 * \fn void chatDestroying(const UserGroup *group)
		 * This slot calls disconnectFromClass"("Chat* chat")" to disconnect
		 * the object from chat "<i>send message</i>" signal.
		 * \param group Users in chat window, which emited the signal.
		 */
		void chatDestroying(const UserGroup *group);

		/*!
		 * \fn void sendRequest(Chat* chat)
		 * Handles chat message send request.
		 * It checks configuration if words fixing is enabled. If it's not,
		 * then the method makes no changes in input text.
		 * \param chat Chat window which sending request is invoked by.
		 */
		void sendRequest(Chat* chat);

	public slots:
		/*!
		 * \fn void onCreateConfig()
		 * Called when config dialog is created.
		 * Connects words fix configuration widgets to slots.
		 */
		void onCreateConfig();

		/*!
		 * \fn onDestroyConfig()
		 * Called when config dialog is going to be destroyed.
		 * Disconnects words fix configuration widgets from slots.
		 */
		void onDestroyConfig();

		/*!
		 * \fn void wordSelected(const QString &word)
		 * Called when a word was selected in configuration dialog
		 * from the words to fix list.
		 * \param word Selected word on the list.
		 */
		void wordSelected(const QString &word);

		/*!
		 * \fn void changeSelected()
		 * Called when "Change selected" button in configuration dialog
		 * was clicked. Changes current replace value for selected word.
		 */
		void changeSelected();

		/*!
		 * \fn void deleteSelected()
		 * Called when "Delete selected" button in configuration dialog
		 * was clicked. Deletes selected word from configuration list
		 * and also from internal words fix list.
		 */
		void deleteSelected();

		/*!
		 * \fn void addNew()
		 * Called when "Add new" button in configuration dialog
		 * was clicked. Adds new word and its fix value from filled fields.
		 */
		void addNew();

		/*!
		 * \fn void moveToNewValue()
		 * Called when return key is pressed in "new word" field
		 * in configuration dialog. Moves input focus to word fix value.
		 */
		void moveToNewValue();
};

#endif
