/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qcheckbox.h>
#include <qcolordialog.h>
#include <qfontdialog.h>
#include <qspinbox.h>
#include <qstringlist.h>
#include <qvbuttongroup.h>

#include "chat.h"
#include "chat_manager.h"
#include "config_file.h"
#include "config_dialog.h"
#include "debug.h"
#include "kadu.h"
#include "kadu_parser.h"
#include "message_box.h"
#include "misc.h"
#include "xosd_notify.h"

extern "C" int xosd_notify_init()
{
	kdebugf();
	xosd_notify = new XOSDNotify(NULL, "xosd_notify");
	kdebugf2();
	return 0;
}

extern "C" void xosd_notify_close()
{
	kdebugf();
	delete xosd_notify;
	xosd_notify = NULL;
	kdebugf2();
}

XOSDNotify::XOSDNotify(QObject *parent, const char *name) : Notifier(parent, name), testXOSD(NULL)
{
	kdebugf();

	optionPrefixes<<"Online"<<"OnlineD"<<"Busy"<<"BusyD"<<"Invisible"<<
					"InvisibleD"<<"Offline"<<"OfflineD"<<"Blocking"<<
					"NewChat"<<"NewMessage"<<"Error"<<"OtherMessage";

	int val;
	CONST_FOREACH(it, optionPrefixes)
	{
		QString tmp = (*it)+"Position";
		config_file.addVariable("XOSD", tmp, 4);
		val = config_file.readNumEntry("XOSD", tmp);
		if (val < 0 || val >= 9) config_file.writeEntry("XOSD", tmp, 4);

		config_file.addVariable("XOSD", (*it)+"FontColor", QColor("blue"));
		config_file.addVariable("XOSD", (*it)+"OutlineColor", QColor("red"));
		config_file.addVariable("XOSD", (*it)+"ShadowColor", QColor("green"));
		config_file.addVariable("XOSD", (*it)+"Font", "-*-courier-*-r-*-*-14-*-*-*-*-*-*-*");

		tmp = (*it)+"Timeout";
		config_file.addVariable("XOSD", tmp, 10);
		val = config_file.readNumEntry("XOSD", tmp);
		if (val < 0 || val >= 2048) config_file.writeEntry("XOSD", tmp, 10);

		tmp = (*it)+"OutlineOffset";
		config_file.addVariable("XOSD", tmp, 0);
		val = config_file.readNumEntry("XOSD", tmp);
		if (val < 0 || val >= 2048) config_file.writeEntry("XOSD", tmp, 0);

		tmp = (*it)+"ShadowOffset";
		config_file.addVariable("XOSD", tmp, 0);
		val = config_file.readNumEntry("XOSD", tmp);
		if (val < 0 || val >= 2048) config_file.writeEntry("XOSD", tmp, 0);
	}
	for(int i = 0; i < 9; ++i)
	{
		config_file.addVariable("XOSD", QString("OffsetX%1").arg(i), 0);
		config_file.addVariable("XOSD", QString("OffsetY%1").arg(i), 0);
	}

	ConfigDialog::addTab(QT_TRANSLATE_NOOP("@default", "XOSD"), dataPath("kadu/modules/data/xosd_notify/xosdblue.png"));

	ConfigDialog::addVGroupBox("XOSD", "XOSD", QT_TRANSLATE_NOOP("@default", "New chat / new message"));
		ConfigDialog::addCheckBox("XOSD", "New chat / new message", QT_TRANSLATE_NOOP("@default", "Show message content"), "ShowContentMessage", true);
		ConfigDialog::addSpinBox("XOSD", "New chat / new message", QT_TRANSLATE_NOOP("@default", "Number of quoted characters"), "CiteSign", 10, 1000, 1, 50, 0, 0, Advanced);

	ConfigDialog::addVGroupBox("XOSD", "XOSD", QT_TRANSLATE_NOOP("@default", "Status change"));
		ConfigDialog::addCheckBox("XOSD", "Status change", QT_TRANSLATE_NOOP("@default", "Add description"), "NotifyWithDescription", true);
		ConfigDialog::addCheckBox("XOSD", "Status change", QT_TRANSLATE_NOOP("@default", "Use custom syntax"), "NotifyUseSyntax", false, 0, 0, Expert);
		ConfigDialog::addLineEdit("XOSD", "Status change", QT_TRANSLATE_NOOP("@default", "Syntax"), "NotifySyntax", QString::null, Kadu::SyntaxText, 0, Expert);

	ConfigDialog::addVGroupBox("XOSD", "XOSD", QT_TRANSLATE_NOOP("@default", "Parameters"), 0, Advanced);
		ConfigDialog::addHBox("XOSD", "Parameters", "top");
			ConfigDialog::addCheckBox("XOSD", "top", QT_TRANSLATE_NOOP("@default", "Set for all"), "SetAll", false);

		ConfigDialog::addHBox("XOSD", "Parameters", "center");
			QStringList options2;
			QStringList values2;
			options2<<tr("Online")<<tr("Online (d.)")<<tr("Busy")<<tr("Busy (d.)")<<
					tr("Invisible")<<tr("Invisible (d.)")<<tr("Offline")<<tr("Offline (d.)")<<
					tr("Blocking")<<tr("New chat")<<tr("New message in chat")<<tr("Error")<<tr("Other message");
			values2<<"0"<<"1"<<"2"<<"3"<<"4"<<"5"<<"6"<<"7"<<"8"<<"9"<<"10"<<"11"<<"12";
			ConfigDialog::addVRadioGroup("XOSD", "center", QT_TRANSLATE_NOOP("@default", "Type"), "LastSelected", options2, values2, "0");

			ConfigDialog::addVBox("XOSD", "center", "bottom");
				ConfigDialog::addLabel("XOSD", "bottom", 0, "stretcher2");
				ConfigDialog::addLabel("XOSD", "bottom", 0, "stretcher3");

				QStringList options, values;
				options<<tr("Top left")<<tr("Top")<<tr("Top right")<<
						tr("Left")<<tr("Center")<<tr("Right")<<
						tr("Bottom left")<<tr("Bottom")<<tr("Bottom right");
				values<<"0"<<"1"<<"2"<<"3"<<"4"<<"5"<<"6"<<"7"<<"8";
				ConfigDialog::addRadioGroup("XOSD", "bottom", QT_TRANSLATE_NOOP("@default", "Positions"), "LastPosition1", options, values, 3, Qt::Horizontal, "4");
				ConfigDialog::addSpinBox("XOSD", "bottom", QT_TRANSLATE_NOOP("@default", "Timeout"), "LastTimeout", 1, 2048, 1, 10);
				ConfigDialog::addSpinBox("XOSD", "bottom", QT_TRANSLATE_NOOP("@default", "Outline offset"), "LastOutline", 0, 2048, 1, 0);
				ConfigDialog::addSpinBox("XOSD", "bottom", QT_TRANSLATE_NOOP("@default", "Shadow offset"), "LastShadow", 0, 2048, 1, 0);
				ConfigDialog::addPushButton("XOSD", "bottom", QT_TRANSLATE_NOOP("@default", "Change font color"));
				ConfigDialog::addPushButton("XOSD", "bottom", QT_TRANSLATE_NOOP("@default", "Change outline color"));
				ConfigDialog::addPushButton("XOSD", "bottom", QT_TRANSLATE_NOOP("@default", "Change shadow color"));
				ConfigDialog::addPushButton("XOSD", "bottom", QT_TRANSLATE_NOOP("@default", "Change font"));

				ConfigDialog::addPushButton("XOSD", "bottom", QT_TRANSLATE_NOOP("@default", "Test"));
	ConfigDialog::addHGroupBox("XOSD", "XOSD", QT_TRANSLATE_NOOP("@default", "Offsets"), 0, Advanced);
		ConfigDialog::addRadioGroup("XOSD", "Offsets", QT_TRANSLATE_NOOP("@default","Positions:"), "LastPosition2", options, values, 3, Qt::Horizontal, "4");
		ConfigDialog::addVBox("XOSD", "Offsets", "xy");
			ConfigDialog::addSpinBox("XOSD", "xy", QT_TRANSLATE_NOOP("@default","X offset"), "LastXOffset", -2048, 2048, 1, 0);
			ConfigDialog::addSpinBox("XOSD", "xy", QT_TRANSLATE_NOOP("@default","Y offset"), "LastYOffset", -2048, 2048, 1, 0);


	ConfigDialog::connectSlot("XOSD", "Show message content", SIGNAL(toggled(bool)), this, SLOT(toggled_ShowMessageContent(bool)));
	ConfigDialog::connectSlot("XOSD", "Use custom syntax", SIGNAL(toggled(bool)), this, SLOT(toggled_UseNotifySyntax(bool)));

	ConfigDialog::connectSlot("XOSD", "Set for all", SIGNAL(toggled(bool)), this, SLOT(toggled_SetAll(bool)));
	ConfigDialog::connectSlot("XOSD", "Type", SIGNAL(clicked(int)), this, SLOT(clicked_Type(int)));

	ConfigDialog::connectSlot("XOSD", "Positions", SIGNAL(clicked(int)), this, SLOT(clicked_Positions(int)));
	ConfigDialog::connectSlot("XOSD", "Timeout", SIGNAL(valueChanged(int)), this, SLOT(changed_Timeout(int)));
	ConfigDialog::connectSlot("XOSD", "Shadow offset", SIGNAL(valueChanged(int)), this, SLOT(changed_ShadowOffset(int)));
	ConfigDialog::connectSlot("XOSD", "Outline offset", SIGNAL(valueChanged(int)), this, SLOT(changed_OutlineOffset(int)));
	ConfigDialog::connectSlot("XOSD", "Change font color", SIGNAL(clicked()), this, SLOT(clicked_ChangeFgColor()));
	ConfigDialog::connectSlot("XOSD", "Change outline color", SIGNAL(clicked()), this, SLOT(clicked_ChangeOutlineColor()));
	ConfigDialog::connectSlot("XOSD", "Change shadow color", SIGNAL(clicked()), this, SLOT(clicked_ChangeShadowColor()));
	ConfigDialog::connectSlot("XOSD", "Change font", SIGNAL(clicked()), this, SLOT(clicked_ChangeFont()));
	ConfigDialog::connectSlot("XOSD", "Test", SIGNAL(clicked()), this, SLOT(clicked_Test()));

	ConfigDialog::connectSlot("XOSD", "Positions:", SIGNAL(clicked(int)), this, SLOT(clicked_Positions2(int)));
	ConfigDialog::connectSlot("XOSD", "X offset", SIGNAL(valueChanged(int)), this, SLOT(changed_XOffset(int)));
	ConfigDialog::connectSlot("XOSD", "Y offset", SIGNAL(valueChanged(int)), this, SLOT(changed_YOffset(int)));

	ConfigDialog::registerSlotOnCreateTab("XOSD", this, SLOT(onCreateConfigDialog()));
	ConfigDialog::registerSlotOnApplyTab("XOSD", this, SLOT(onApplyConfigDialog()));
	ConfigDialog::registerSlotOnCloseTab("XOSD", this, SLOT(onCloseConfigDialog()));

	QMap<QString, QString> s;
	s["NewChat"]=SLOT(newChat(Protocol *, UserListElements, const QString &, time_t));
	s["NewMessage"]=SLOT(newMessage(Protocol *, UserListElements, const QString &, time_t, bool &));
	s["ConnError"]=SLOT(connectionError(Protocol *, const QString &));
	s["toAvailable"]=SLOT(userChangedStatusToAvailable(const QString &, UserListElement));
	s["toBusy"]=SLOT(userChangedStatusToBusy(const QString &, UserListElement));
	s["toInvisible"]=SLOT(userChangedStatusToInvisible(const QString &, UserListElement));
	s["toNotAvailable"]=SLOT(userChangedStatusToNotAvailable(const QString &, UserListElement));

//	s["StatusChanged"] = SLOT(userStatusChanged(UserListElement, QString, const UserStatus &));
	s["Message"] = SLOT(message(const QString &, const QString &, const QMap<QString, QVariant> *, const UserListElement *));

	config_file.addVariable("Notify", "NewChat_XOSD", false);
	config_file.addVariable("Notify", "NewMessage_XOSD", false);
	config_file.addVariable("Notify", "ConnError_XOSD", false);
	config_file.addVariable("Notify", "ChangingStatus_XOSD", false);
	config_file.addVariable("Notify", "toAvailable_XOSD", false);
	config_file.addVariable("Notify", "toBusy_XOSD", false);
	config_file.addVariable("Notify", "toInvisible_XOSD", false);
	config_file.addVariable("Notify", "toNotAvailable_XOSD", false);
	config_file.addVariable("Notify", "Message_XOSD", false);

	notify->registerNotifier(QT_TRANSLATE_NOOP("@default", "XOSD"), this, s);

	timer = new QTimer(this, "timer");
	connect(timer, SIGNAL(timeout()), this, SLOT(oneSecond()));
	kdebugf2();
}

XOSDNotify::~XOSDNotify()
{
	kdebugf();

	disconnect(timer, SIGNAL(timeout()), this, SLOT(oneSecond()));
	delete timer;
	notify->unregisterNotifier("XOSD");

	ConfigDialog::unregisterSlotOnCreateTab("XOSD", this, SLOT(onCreateConfigDialog()));
	ConfigDialog::unregisterSlotOnApplyTab("XOSD", this, SLOT(onApplyConfigDialog()));
	ConfigDialog::unregisterSlotOnCloseTab("XOSD", this, SLOT(onCloseConfigDialog()));

	ConfigDialog::disconnectSlot("XOSD", "Show message content", SIGNAL(toggled(bool)), this, SLOT(toggled_ShowMessageContent(bool)));
	ConfigDialog::disconnectSlot("XOSD", "Use custom syntax", SIGNAL(toggled(bool)), this, SLOT(toggled_UseNotifySyntax(bool)));

	ConfigDialog::disconnectSlot("XOSD", "Set for all", SIGNAL(toggled(bool)), this, SLOT(toggled_SetAll(bool)));
	ConfigDialog::disconnectSlot("XOSD", "Type", SIGNAL(clicked(int)), this, SLOT(clicked_Type(int)));

	ConfigDialog::disconnectSlot("XOSD", "Positions", SIGNAL(clicked(int)), this, SLOT(clicked_Positions(int)));
	ConfigDialog::disconnectSlot("XOSD", "Timeout", SIGNAL(valueChanged(int)), this, SLOT(changed_Timeout(int)));
	ConfigDialog::disconnectSlot("XOSD", "Shadow offset", SIGNAL(valueChanged(int)), this, SLOT(changed_ShadowOffset(int)));
	ConfigDialog::disconnectSlot("XOSD", "Outline offset", SIGNAL(valueChanged(int)), this, SLOT(changed_OutlineOffset(int)));
	ConfigDialog::disconnectSlot("XOSD", "Change font color", SIGNAL(clicked()), this, SLOT(clicked_ChangeFgColor()));
	ConfigDialog::disconnectSlot("XOSD", "Change outline color", SIGNAL(clicked()), this, SLOT(clicked_ChangeOutlineColor()));
	ConfigDialog::disconnectSlot("XOSD", "Change shadow color", SIGNAL(clicked()), this, SLOT(clicked_ChangeShadowColor()));
	ConfigDialog::disconnectSlot("XOSD", "Change font", SIGNAL(clicked()), this, SLOT(clicked_ChangeFont()));
	ConfigDialog::disconnectSlot("XOSD", "Test", SIGNAL(clicked()), this, SLOT(clicked_Test()));

	ConfigDialog::disconnectSlot("XOSD", "Positions:", SIGNAL(clicked(int)), this, SLOT(clicked_Positions2(int)));
	ConfigDialog::disconnectSlot("XOSD", "X offset", SIGNAL(valueChanged(int)), this, SLOT(changed_XOffset(int)));
	ConfigDialog::disconnectSlot("XOSD", "Y offset", SIGNAL(valueChanged(int)), this, SLOT(changed_YOffset(int)));

	ConfigDialog::removeControl("XOSD", "X offset");
	ConfigDialog::removeControl("XOSD", "Y offset");
	ConfigDialog::removeControl("XOSD", "xy");
	ConfigDialog::removeControl("XOSD", "Positions:");
	ConfigDialog::removeControl("XOSD", "Offsets");
	ConfigDialog::removeControl("XOSD", "Test");
	ConfigDialog::removeControl("XOSD", "Change font");
	ConfigDialog::removeControl("XOSD", "Change shadow color");
	ConfigDialog::removeControl("XOSD", "Change outline color");
	ConfigDialog::removeControl("XOSD", "Change font color");
	ConfigDialog::removeControl("XOSD", "Outline offset");
	ConfigDialog::removeControl("XOSD", "Shadow offset");
	ConfigDialog::removeControl("XOSD", "Timeout");
	ConfigDialog::removeControl("XOSD", "Positions");
	ConfigDialog::removeControl("XOSD", 0, "stretcher3");
	ConfigDialog::removeControl("XOSD", 0, "stretcher2");
	ConfigDialog::removeControl("XOSD", "bottom");
	ConfigDialog::removeControl("XOSD", "Type");
	ConfigDialog::removeControl("XOSD", "center");
	ConfigDialog::removeControl("XOSD", "Set for all");
	ConfigDialog::removeControl("XOSD", "top");
	ConfigDialog::removeControl("XOSD", "Parameters");

	ConfigDialog::removeControl("XOSD", "Syntax");
	ConfigDialog::removeControl("XOSD", "Use custom syntax");
	ConfigDialog::removeControl("XOSD", "Add description");
	ConfigDialog::removeControl("XOSD", "Status change");

	ConfigDialog::removeControl("XOSD", "Number of quoted characters");
	ConfigDialog::removeControl("XOSD", "Show message content");
	ConfigDialog::removeControl("XOSD", "New chat / new message");

	ConfigDialog::removeTab("XOSD");

	kdebugf2();
}

void XOSDNotify::position2PosAlign(int position, xosd_pos &pos, xosd_align &align)
{
	switch (position % 3)
	{
		case 0: align = XOSD_left; break;
		case 1: align = XOSD_center; break;
		case 2: align = XOSD_right; break;
	};
	switch (position / 3)
	{
		case 0: pos = XOSD_top; break;
		case 2: pos = XOSD_bottom; break;
		case 1:
		default:
				pos = XOSD_middle; break;
	};
}

void XOSDNotify::test(const QString &text, const struct TestConfig &config)
{
	kdebugf();
	if (testXOSD)
		destroyTest();

	testXOSD = xosd_create(1);

	xosd_pos pos;
	xosd_align align;
	position2PosAlign(config.position, pos, align);

	xosd_set_pos(testXOSD, pos);
	xosd_set_align(testXOSD, align);

	if (!config.font.isEmpty())
		xosd_set_font(testXOSD, config.font.local8Bit().data());
	xosd_set_colour(testXOSD, config.font_color.name().local8Bit().data());
	xosd_set_shadow_colour(testXOSD, config.shadow_color.name().local8Bit().data());
	xosd_set_outline_colour(testXOSD, config.outline_color.name().local8Bit().data());

	xosd_set_shadow_offset(testXOSD, config.shadow_offset);
	xosd_set_outline_offset(testXOSD, config.outline_offset);

	if (config.position >= 0 && config.position <= 8)
	{
		xosd_set_horizontal_offset(testXOSD, offsets[config.position].x);
		xosd_set_vertical_offset(testXOSD, offsets[config.position].y);
	}
	else
		kdebugm(KDEBUG_WARNING, "config.position: %d\n", config.position);

	xosd_display(testXOSD, 0, XOSD_string, text.local8Bit().data());

	QTimer::singleShot(1000*config.timeout, this, SLOT(destroyTest()));

	kdebugf2();
}

void XOSDNotify::addLine(int position, const QString &text, int timeout,
					const QString &font, const QColor &font_color,
					const QColor &shadow_color, int shadow_offset,
					const QColor &outline_color, int outline_offset)
{
	kdebugf();

	OSDLine line;
	if (position < 0 || position >= 9)
	{
		kdebugmf(KDEBUG_FUNCTION_END|KDEBUG_WARNING, "end: position=%d\n", position);
		return;
	}
	if (timeout <= 0 || timeout > 2048)
	{
		kdebugmf(KDEBUG_FUNCTION_END|KDEBUG_WARNING, "end: timeout: %d<=0\n", timeout);
		return;
	}
	line.font_size = getFontSize(font);
	if (line.font_size <= 0)
	{
		kdebugmf(KDEBUG_FUNCTION_END|KDEBUG_WARNING, "end: font_size: %d<=0\n", timeout);
		return;
	}

	line.handle = xosd_create(1);
	line.text = text;
	line.timeout = timeout;
	line.font_color = font_color;
	line.outline_color = font_color;
	line.shadow_color = shadow_color;
	line.font = font;

	xosd_pos pos;
	xosd_align align;
	position2PosAlign(position, pos, align);

	xosd_set_pos(line.handle, pos);
	xosd_set_align(line.handle, align);

	if (!font.isEmpty())
		xosd_set_font(line.handle, font.local8Bit().data());
	if (font_color.isValid())
		xosd_set_colour(line.handle, font_color.name().local8Bit().data());
	if (shadow_color.isValid())
	{
		xosd_set_shadow_colour(line.handle, shadow_color.name().local8Bit().data());
		xosd_set_shadow_offset(line.handle, shadow_offset);
	}
	if (outline_color.isValid())
	{
		xosd_set_outline_colour(line.handle, outline_color.name().local8Bit().data());
		xosd_set_outline_offset(line.handle, outline_offset);
	}

	int x_offset = config_file.readNumEntry("XOSD", QString("OffsetX%1").arg(position));
	int y_offset = config_file.readNumEntry("XOSD", QString("OffsetY%1").arg(position));

	CONST_FOREACH(line, lines[position])
		y_offset += (*line).font_size + 1;

	xosd_set_horizontal_offset(line.handle, x_offset);
	xosd_set_vertical_offset(line.handle, y_offset);

	lines[position].append(line);
	xosd_display(line.handle, 0, XOSD_string, text.local8Bit().data());

	if (!timer->isActive())
		timer->start(1000);

	kdebugf2();
}

void XOSDNotify::refresh(int position)
{
	kdebugf();

	if (position < 0 || position >= 9)
	{
		kdebugmf(KDEBUG_FUNCTION_END|KDEBUG_WARNING, "end: position: %d<0 || %d>=9\n", position, position);
		return;
	}

	int y_offset = config_file.readNumEntry("XOSD", QString("OffsetY%1").arg(position));
	CONST_FOREACH(line, lines[position])
	{
		xosd_set_vertical_offset((*line).handle, y_offset);
		xosd_hide((*line).handle);
		xosd_show((*line).handle);
		y_offset += (*line).font_size + 1;
	}

	kdebugf2();
}

void XOSDNotify::oneSecond()
{
	kdebugf();

	int sum = 0;
	for (int k = 0; k < 9; ++k)
	{
		QValueList<OSDLine> &linesK = lines[k];

		const unsigned int count = linesK.count();
//		unsigned int i = count - 1;
		bool need_refresh = false;
		if (count == 0)
			continue;

		QValueListIterator<OSDLine> line = linesK.end();
		--line;
		for (unsigned int j = 0; j < count; ++j, --line)
			if ((*line).timeout-- < 0)
			{
				xosd_destroy((*line).handle);
				line = linesK.remove(line);
				need_refresh = true;
			}
		if (need_refresh)
			refresh(k);
		sum += count;
	}

	if (sum == 0)
		timer->stop();

	kdebugf2();
}

void XOSDNotify::destroyTest()
{
	kdebugf();

	if (testXOSD == NULL)
	{
		kdebugf2();
		return;
	}
	xosd_destroy(testXOSD);
	testXOSD = NULL;

	kdebugf2();
}

void XOSDNotify::onCreateConfigDialog()
{
	kdebugf();

	toggled_ShowMessageContent(config_file.readBoolEntry("XOSD", "ShowContentMessage"));
	toggled_UseNotifySyntax(config_file.readBoolEntry("XOSD", "NotifyUseSyntax"));
	toggled_SetAll(config_file.readBoolEntry("XOSD", "SetAll"));

	configs.clear();
	CONST_FOREACH(prefix, optionPrefixes)
	{
		TestConfig c;
		c.type = *prefix;
		c.position = config_file.readNumEntry("XOSD", (*prefix)+"Position");
		c.font_color = config_file.readColorEntry("XOSD", (*prefix)+"FontColor");
		c.outline_color = config_file.readColorEntry("XOSD", (*prefix)+"OutlineColor");
		c.shadow_color = config_file.readColorEntry("XOSD", (*prefix)+"ShadowColor");
		c.font = config_file.readEntry("XOSD", (*prefix)+"Font");
		c.font_size = getFontSize(c.font);
		c.timeout = config_file.readNumEntry("XOSD", (*prefix)+"Timeout");
		c.shadow_offset = config_file.readNumEntry("XOSD", (*prefix)+"ShadowOffset");
		c.outline_offset = config_file.readNumEntry("XOSD", (*prefix)+"OutlineOffset");
		configs[c.type] = c;
	}
	for(int i = 0; i < 9; ++i)
	{
		offsets[i].x = config_file.readNumEntry("XOSD", QString("OffsetX%1").arg(i));
		offsets[i].y = config_file.readNumEntry("XOSD", QString("OffsetY%1").arg(i));
	}

	currentOptionPrefix = optionPrefixes[config_file.readNumEntry("XOSD", "LastSelected")];
	main_xlfd_chooser = new XLFDChooser(this, "main_xlfd_chooser");

	kdebugf2();
}

void XOSDNotify::onCloseConfigDialog()
{
	kdebugf();

	configs.clear();
	main_xlfd_chooser->deleteLater();
	main_xlfd_chooser = NULL;

	kdebugf2();
}

void XOSDNotify::onApplyConfigDialog()
{
	kdebugf();

	CONST_FOREACH(prefix, optionPrefixes)
	{
		TestConfig c = configs[*prefix];
		config_file.writeEntry("XOSD", (*prefix)+"Position", c.position);
		config_file.writeEntry("XOSD", (*prefix)+"FontColor", c.font_color);
		config_file.writeEntry("XOSD", (*prefix)+"OutlineColor", c.outline_color);
		config_file.writeEntry("XOSD", (*prefix)+"ShadowColor", c.shadow_color);
		config_file.writeEntry("XOSD", (*prefix)+"Font", c.font);
		config_file.writeEntry("XOSD", (*prefix)+"Timeout", c.timeout);
		config_file.writeEntry("XOSD", (*prefix)+"ShadowOffset", c.shadow_offset);
		config_file.writeEntry("XOSD", (*prefix)+"OutlineOffset", c.outline_offset);
	}
	for(int i = 0; i < 9; ++i)
	{
		config_file.writeEntry("XOSD", QString("OffsetX%1").arg(i), offsets[i].x);
		config_file.writeEntry("XOSD", QString("OffsetY%1").arg(i), offsets[i].y);
	}

	kdebugf2();
}

void XOSDNotify::toggled_SetAll(bool val)
{
	kdebugf();
	ConfigDialog::getVButtonGroup("XOSD", "Type")->setEnabled(!val);
	kdebugf2();
}

void XOSDNotify::clicked_Type(int id)
{
	kdebugf();

	if (optionPrefixes[id] == currentOptionPrefix)
		return;
	currentOptionPrefix = optionPrefixes[id];
	TestConfig &conf = configs[currentOptionPrefix];

	ConfigDialog::getButtonGroup("XOSD", "Positions")->setButton(conf.position);
	ConfigDialog::getSpinBox("XOSD", "Timeout")->setValue(conf.timeout);
	ConfigDialog::getSpinBox("XOSD", "Shadow offset")->setValue(conf.shadow_offset);
	ConfigDialog::getSpinBox("XOSD", "Outline offset")->setValue(conf.outline_offset);

	kdebugf2();
}

void XOSDNotify::clicked_Positions(int pos)
{
	kdebugf();

	if (ConfigDialog::getCheckBox("XOSD", "Set for all")->isChecked())
		FOREACH(config, configs)
			config.data().position = pos;
	else
		configs[currentOptionPrefix].position = pos;

	kdebugf2();
}

void XOSDNotify::changed_Timeout(int value)
{
	kdebugf();

	if (ConfigDialog::getCheckBox("XOSD", "Set for all")->isChecked())
		FOREACH(config, configs)
			config.data().timeout = value;
	else
		configs[currentOptionPrefix].timeout = value;

	kdebugf2();
}

void XOSDNotify::changed_OutlineOffset(int value)
{
	kdebugf();

	if (ConfigDialog::getCheckBox("XOSD", "Set for all")->isChecked())
		FOREACH(config, configs)
			config.data().outline_offset = value;
	else
		configs[currentOptionPrefix].outline_offset=value;

	kdebugf2();
}

void XOSDNotify::changed_ShadowOffset(int value)
{
	kdebugf();

	if (ConfigDialog::getCheckBox("XOSD", "Set for all")->isChecked())
		FOREACH(config, configs)
			config.data().shadow_offset = value;
	else
		configs[currentOptionPrefix].shadow_offset = value;

	kdebugf2();
}

void XOSDNotify::clicked_ChangeFgColor()
{
	kdebugf();

	QColor color = QColorDialog::getColor(configs[currentOptionPrefix].font_color, 0, "Color dialog");
	if (color.isValid())
	{
		if (ConfigDialog::getCheckBox("XOSD", "Set for all")->isChecked())
			FOREACH(config, configs)
				config.data().font_color = color;
		else
			configs[currentOptionPrefix].font_color = color;
	}

	kdebugf2();
}

void XOSDNotify::clicked_ChangeOutlineColor()
{
	kdebugf();

	QColor color = QColorDialog::getColor(configs[currentOptionPrefix].outline_color, 0, "Color dialog");
	if (color.isValid())
	{
		if (ConfigDialog::getCheckBox("XOSD", "Set for all")->isChecked())
			FOREACH(config, configs)
				config.data().outline_color = color;
		else
			configs[currentOptionPrefix].outline_color = color;
	}

	kdebugf2();
}

void XOSDNotify::clicked_ChangeShadowColor()
{
	kdebugf();

	QColor color = QColorDialog::getColor(configs[currentOptionPrefix].shadow_color, 0, "Color dialog");
	if (color.isValid())
	{
		if (ConfigDialog::getCheckBox("XOSD", "Set for all")->isChecked())
			FOREACH(config, configs)
				config.data().shadow_color = color;
		else
			configs[currentOptionPrefix].shadow_color = color;
	}

	kdebugf2();
}

void XOSDNotify::clicked_ChangeFont()
{
	kdebugf();
	//configs[currentOptionPrefix].font
	main_xlfd_chooser->getFont(this, SLOT(fontSelected(const QString &)), configs[currentOptionPrefix].font);

/*	bool ok;

	QFont font;
	font.setRawMode(true);
	font.setRawName(configs[currentOptionPrefix].font);
	font.setRawMode(true);

	kdebugm(KDEBUG_INFO, "%s\n", font.rawName().local8Bit().data());
	font= QFontDialog::getFont(&ok, font, 0);
	font.setRawMode(true);
	kdebugm(KDEBUG_WARNING, "%s\n", font.rawName().local8Bit().data());
	if (ok)
	{
		if (ConfigDialog::getCheckBox("XOSD", "Set for all")->isChecked())
			for (QMap<QString, TestConfig>::iterator it=configs.begin();
				it!=configs.end();
				++it)
				it.data().font=font.rawName();
		else
			configs[currentOptionPrefix].font = font.rawName();
	}
*/
	kdebugf2();
}

int XOSDNotify::getFontSize(const QString &xlfd)
{
	kdebugf();

	QStringList font = QStringList::split("-", xlfd, true);
	if (font.size() < 8)
	{
		kdebugf2();
		return -1;
	}
	else if (font[7] == "*" || font[7].isEmpty())
	{
		kdebugf2();
		return -1;
	}
	else
	{
		kdebugf2();
		return font[7].toInt();
	}
}

void XOSDNotify::fontSelected(const QString &font)
{
	kdebugf();
	kdebugm(KDEBUG_INFO, "%s %d %d\n", font.local8Bit().data(), font.isEmpty(), font.isNull());

	int size = getFontSize(font);
	if (size < 0)
	{
		MessageBox::msg(tr("Please select font size! (pxlsz)"));
		kdebugf2();
		return;
	}
	if (ConfigDialog::getCheckBox("XOSD", "Set for all")->isChecked())
		FOREACH(config, configs)
		{
			config.data().font = font;
			config.data().font_size = size;
		}
	else
	{
		configs[currentOptionPrefix].font = font;
		configs[currentOptionPrefix].font_size = size;
	}

	kdebugf2();
}

void XOSDNotify::clicked_Test()
{
	kdebugf();
	test(tr("Testing configuration"), configs[currentOptionPrefix]);
	kdebugf2();
}

void XOSDNotify::toggled_ShowMessageContent(bool val)
{
	kdebugf();
	ConfigDialog::getSpinBox("XOSD", "Number of quoted characters")->setEnabled(val);
	kdebugf2();
}

void XOSDNotify::toggled_UseNotifySyntax(bool val)
{
	kdebugf();
	ConfigDialog::getLineEdit("XOSD", "Syntax")->setEnabled(val);
	kdebugf2();
}

void XOSDNotify::clicked_Positions2(int id)
{
	kdebugf();
	ConfigDialog::getSpinBox("XOSD", "X offset")->setValue(offsets[id].x);
	ConfigDialog::getSpinBox("XOSD", "Y offset")->setValue(offsets[id].y);
	kdebugf2();
}

void XOSDNotify::changed_XOffset(int value)
{
	kdebugf();
	QButtonGroup *group = ConfigDialog::getButtonGroup("XOSD", "Positions:");
	offsets[group->id(group->selected())].x = value;
	kdebugf2();
}

void XOSDNotify::changed_YOffset(int value)
{
	kdebugf();
	QButtonGroup *group = ConfigDialog::getButtonGroup("XOSD", "Positions:");
	offsets[group->id(group->selected())].y = value;
	kdebugf2();
}

void XOSDNotify::newChat(Protocol *protocol, UserListElements senders, const QString &msg, time_t /*t*/)
{
	kdebugf();

	QString text;
	if (config_file.readBoolEntry("XOSD", "ShowContentMessage"))
	{
		unsigned int citeSign = config_file.readUnsignedNumEntry("XOSD","CiteSign");
		QString cite = toPlainText(msg);
		if (cite.length() > citeSign)
			cite = cite.left(citeSign)+"...";
		text = narg(tr("Chat with %1: %2"), senders[0].altNick(), cite);
	}
	else
		text = tr("Chat with %1").arg(senders[0].altNick());


	addLine(config_file.readNumEntry("XOSD",	"NewChatPosition"),
			text,
			config_file.readNumEntry("XOSD",	"NewChatTimeout"),
			config_file.readEntry("XOSD",		"NewChatFont"),
			config_file.readColorEntry("XOSD",	"NewChatFontColor"),
			config_file.readColorEntry("XOSD",	"NewChatShadowColor"),
			config_file.readNumEntry("XOSD",	"NewChatShadowOffset"),
			config_file.readColorEntry("XOSD",	"NewChatOutlineColor"),
			config_file.readNumEntry("XOSD",	"NewChatOutlineOffset"));

	kdebugf2();
}

void XOSDNotify::newMessage(Protocol *protocol, UserListElements senders, const QString &msg, time_t /*t*/, bool &/*grab*/)
{
	kdebugf();

	Chat* chat = chat_manager->findChat(senders);
	if (chat == NULL)
	{
		kdebugmf(KDEBUG_ERROR|KDEBUG_FUNCTION_END, "end: chat==NULL!\n");
		return;
	}

	if (!chat->isActiveWindow())
	{
		QString text;
		if (config_file.readBoolEntry("XOSD", "ShowContentMessage"))
		{
			unsigned int citeSign = config_file.readUnsignedNumEntry("XOSD","CiteSign");
			QString cite = toPlainText(msg);
			if (msg.length() > citeSign)
				cite = cite.left(citeSign)+"...";
			text = narg(tr("New message from %1: %2"), senders[0].altNick(), cite);
		}
		else
			text = tr("New message from %1").arg(senders[0].altNick());
		addLine(config_file.readNumEntry("XOSD",	"NewMessagePosition"),
				text,
				config_file.readNumEntry("XOSD",	"NewMessageTimeout"),
				config_file.readEntry("XOSD",		"NewMessageFont"),
				config_file.readColorEntry("XOSD",	"NewMessageFontColor"),
				config_file.readColorEntry("XOSD",	"NewMessageShadowColor"),
				config_file.readNumEntry("XOSD",	"NewMessageShadowOffset"),
				config_file.readColorEntry("XOSD",	"NewMessageOutlineColor"),
				config_file.readNumEntry("XOSD",	"NewMessageOutlineOffset"));
	}

	kdebugf2();
}

void XOSDNotify::connectionError(Protocol *, const QString &message)
{
	kdebugf();

	addLine(config_file.readNumEntry("XOSD",	"ErrorPosition"),
			tr("Error: %1").arg(message),
			config_file.readNumEntry("XOSD",	"ErrorTimeout"),
			config_file.readEntry("XOSD",		"ErrorFont"),
			config_file.readColorEntry("XOSD",	"ErrorFontColor"),
			config_file.readColorEntry("XOSD",	"ErrorShadowColor"),
			config_file.readNumEntry("XOSD",	"ErrorShadowOffset"),
			config_file.readColorEntry("XOSD",	"ErrorOutlineColor"),
			config_file.readNumEntry("XOSD",	"ErrorOutlineOffset"));

	kdebugf2();
}

void XOSDNotify::userStatusChanged(UserListElement /*ule*/, QString /*protocolName*/, const UserStatus &/*oldStatus*/)
{
	kdebugf();
	kdebugf2();
}

void XOSDNotify::userChangedStatusToAvailable(const QString &protocolName, UserListElement ule)
{
	kdebugf();

	QString statusName = ule.status(protocolName).hasDescription() ? "OnlineD" : "Online";
	QString text;
	if (config_file.readBoolEntry("XOSD", "NotifyUseSyntax"))
		text = KaduParser::parse(config_file.readEntry("XOSD", "NotifySyntax"), ule, true);
	else if (ule.status(protocolName).hasDescription() && config_file.readBoolEntry("XOSD","NotifyWithDescription"))
		text = narg(tr("%1 is available with description: %2"),
				ule.altNick(), ule.status(protocolName).description());
	else
		text = tr("%1 is available").arg(ule.altNick());

	addLine(config_file.readNumEntry("XOSD", statusName+"Position"),
			text,
			config_file.readNumEntry("XOSD",	statusName+"Timeout"),
			config_file.readEntry("XOSD",		statusName+"Font"),
			config_file.readColorEntry("XOSD",	statusName+"FontColor"),
			config_file.readColorEntry("XOSD",	statusName+"ShadowColor"),
			config_file.readNumEntry("XOSD",	statusName+"ShadowOffset"),
			config_file.readColorEntry("XOSD",	statusName+"OutlineColor"),
			config_file.readNumEntry("XOSD",	statusName+"OutlineOffset"));

	kdebugf2();
}

void XOSDNotify::userChangedStatusToBusy(const QString &protocolName, UserListElement ule)
{
	kdebugf();

	QString statusName = ule.status(protocolName).hasDescription() ? "BusyD" : "Busy";
	QString text;
	if (config_file.readBoolEntry("XOSD", "NotifyUseSyntax"))
		text = KaduParser::parse(config_file.readEntry("XOSD", "NotifySyntax"), ule, true);
	else if (ule.status(protocolName).hasDescription() && config_file.readBoolEntry("XOSD","NotifyWithDescription"))
		text = narg(tr("%1 is busy with description: %2"),
				ule.altNick(), ule.status(protocolName).description());
	else
		text = tr("%1 is busy").arg(ule.altNick());

	addLine(config_file.readNumEntry("XOSD", statusName+"Position"),
			text,
			config_file.readNumEntry("XOSD",	statusName+"Timeout"),
			config_file.readEntry("XOSD",		statusName+"Font"),
			config_file.readColorEntry("XOSD",	statusName+"FontColor"),
			config_file.readColorEntry("XOSD",	statusName+"ShadowColor"),
			config_file.readNumEntry("XOSD",	statusName+"ShadowOffset"),
			config_file.readColorEntry("XOSD",	statusName+"OutlineColor"),
			config_file.readNumEntry("XOSD",	statusName+"OutlineOffset"));

	kdebugf2();
}

void XOSDNotify::userChangedStatusToInvisible(const QString &protocolName, UserListElement ule)
{
	kdebugf();

	QString statusName = ule.status(protocolName).hasDescription() ? "InvisibleD" : "Invisible";
	QString text;
	if (config_file.readBoolEntry("XOSD", "NotifyUseSyntax"))
		text = KaduParser::parse(config_file.readEntry("XOSD", "NotifySyntax"), ule, true);
	else if (ule.status(protocolName).hasDescription() && config_file.readBoolEntry("XOSD","NotifyWithDescription"))
		text = narg(tr("%1 is invisible with description: %2"),
				ule.altNick(), ule.status(protocolName).description());
	else
		text = tr("%1 is invisible").arg(ule.altNick());

	addLine(config_file.readNumEntry("XOSD", statusName+"Position"),
			text,
			config_file.readNumEntry("XOSD",	statusName+"Timeout"),
			config_file.readEntry("XOSD",		statusName+"Font"),
			config_file.readColorEntry("XOSD",	statusName+"FontColor"),
			config_file.readColorEntry("XOSD",	statusName+"ShadowColor"),
			config_file.readNumEntry("XOSD",	statusName+"ShadowOffset"),
			config_file.readColorEntry("XOSD",	statusName+"OutlineColor"),
			config_file.readNumEntry("XOSD",	statusName+"OutlineOffset"));

	kdebugf2();
}

void XOSDNotify::userChangedStatusToNotAvailable(const QString &protocolName, UserListElement ule)
{
	kdebugf();

	QString statusName = ule.status(protocolName).hasDescription() ? "OfflineD" : "Offline";
	QString text;
	if (config_file.readBoolEntry("XOSD", "NotifyUseSyntax"))
		text = KaduParser::parse(config_file.readEntry("XOSD", "NotifySyntax"), ule, true);
	else if (ule.status(protocolName).hasDescription() && config_file.readBoolEntry("XOSD","NotifyWithDescription"))
		text = narg(tr("%1 is not available with description: %2"),
				ule.altNick(), ule.status(protocolName).description());
	else
		text = tr("%1 is not available").arg(ule.altNick());

	addLine(config_file.readNumEntry("XOSD", statusName+"Position"),
			text,
			config_file.readNumEntry("XOSD",	statusName+"Timeout"),
			config_file.readEntry("XOSD",		statusName+"Font"),
			config_file.readColorEntry("XOSD",	statusName+"FontColor"),
			config_file.readColorEntry("XOSD",	statusName+"ShadowColor"),
			config_file.readNumEntry("XOSD",	statusName+"ShadowOffset"),
			config_file.readColorEntry("XOSD",	statusName+"OutlineColor"),
			config_file.readNumEntry("XOSD",	statusName+"OutlineOffset"));

	kdebugf2();
}

void XOSDNotify::message(const QString &from, const QString &message, const QMap<QString, QVariant> *parameters, const UserListElement * /*ule*/)
{
	kdebugf();

	int position = -1, timeout = -1, shadow_offset = -1, outline_offset = -1;
	QString font;
	QColor font_color, shadow_color, outline_color;
	bool ok1 = false, ok2 = false, ok3 = false, ok4 = false;
	bool showSource = true;
	if (parameters != NULL)
	{
		QMap<QString, QVariant>::const_iterator end = (*parameters).end();

		position = (*parameters)["Position"].toInt(&ok1);
		timeout = (*parameters)["Timeout"].toInt(&ok2);

		font = (*parameters)["Font"].toString();
		font_color = (*parameters)["FontColor"].toColor();

		shadow_color = (*parameters)["ShadowColor"].toColor();
		shadow_offset = (*parameters)["ShadowOffset"].toInt(&ok3);
		outline_color = (*parameters)["OutlineColor"].toColor();
		outline_offset = (*parameters)["OutlineOffset"].toInt(&ok4);

		QMap<QString, QVariant>::const_iterator sit = (*parameters).find("ShowSource");
		if (sit != end)
			showSource = sit.data().toBool();
	}
	if (position < 0 || position >= 9 || !ok1)
		position = config_file.readNumEntry("XOSD", "OtherMessagePosition");
	if (timeout <= 0 || !ok2)
		timeout = config_file.readNumEntry("XOSD", "OtherMessageTimeout");
	if (font.isEmpty())
		font = config_file.readEntry("XOSD", "OtherMessageFont");
	if (!font_color.isValid())
		font_color = config_file.readColorEntry("XOSD", "OtherMessageFontColor");
	if (!shadow_color.isValid())
		shadow_color = config_file.readColorEntry("XOSD", "OtherMessageShadowColor");
	if (shadow_offset < 0 || !ok3)
		shadow_offset = config_file.readNumEntry("XOSD", "OtherMessageShadowOffset");
	if (!outline_color.isValid())
		outline_color = config_file.readColorEntry("XOSD", "OtherMessageOutlineColor");
	if (outline_offset < 0 || !ok4)
		outline_offset = config_file.readNumEntry("XOSD", "OtherMessageOutlineOffset");
	QString text = message;
	text = text.replace("\n", " ");
	if (!from.isEmpty() && showSource)
		text = narg(tr("From %1: %2"), from, text);
	else
		text = QString("%1").arg(text);
	addLine(position, text, timeout, font, font_color, shadow_color, shadow_offset, outline_color, outline_offset);

	kdebugf2();
}

void XOSDNotify::externalEvent(const QString &notifyType, const QString &msg, const UserListElements &ules)
{
}

XLFDChooser *main_xlfd_chooser = NULL;
XOSDNotify *xosd_notify = NULL;
