//                       -*- mode: C++ -*-
//
// Copyright(C) 2005,2007 Stefan Siegl <stesie@brokenpipe.de>
// Copyright(C) 2007 Christian Dietrich <stettberger@brokenpipe.de>
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <iostream>

#include "silcaccount.h"
#include "silccontact.h"
#include "silcmessagemanager.h"

#include <kopetemetacontact.h>
#include <kopetechatsession.h>
#include <kopetechatsessionmanager.h>
#include <kopetemessage.h>

#include <ui/kopeteview.h>

#include <kmimemagic.h>
#include <kdebug.h>
#include <klocale.h>

#include <qfile.h>
#include <qfileinfo.h>
#include <qimage.h>

SilcContact::SilcContact(SilcAccount *account, const QString &nickname, 
			 Kopete::MetaContact *meta, const QString &icon)
  : Kopete::Contact(account, nickname, meta, icon),
    _chatSession(NULL),
    _joined(false)
{
  SilcProtocol *proto = static_cast<SilcProtocol *>(account->protocol());
  setOnlineStatus(proto->statusOffline);
  mimeasm = NULL;
}
 
SilcContact::~SilcContact() 
{
  if(mimeasm)
    SilcTK::silc_mime_assembler_free(mimeasm);
}

SilcAccount *
SilcContact::account(void)
{
  return static_cast<SilcAccount *>(Kopete::Contact::account());
}

Kopete::ChatSession *
SilcContact::manager(Kopete::Contact::CanCreateFlags flags)
{
  if(_chatSession) 
    return _chatSession;

  if(! (flags & Kopete::Contact::CanCreate))
    return NULL; // not allowed to create new chat session

  if(! account()->isConnected())
    return NULL; // account has no SilcEngine attacted, i.e. it's offline
                 // FIXME, shall we connect here?

  QPtrList<Kopete::Contact> myself;
  myself.append(this);
  
  _chatSession = account()->chatSession(myself);
  _chatSession->setDisplayName(nickName());

  // connect the neccessary signals ...
  QObject::connect(manager(), SIGNAL(messageSent(Kopete::Message &, 
						 Kopete::ChatSession *)),
		   this, SLOT(slotSendMessage(Kopete::Message &,
					      Kopete::ChatSession *)));

  QObject::connect(manager(), SIGNAL(closing(Kopete::ChatSession *)), 
		   this, SLOT(slotCloseSession()));

  if(! _joined)
    join();
  
  return _chatSession;
}

KopeteView *
SilcContact::view(void)
{
  Kopete::ChatSession *session = manager(Kopete::Contact::CanCreate);
  KopeteView *view = session->view(true);

  if(! view) {
    std::cerr << "Dude, FAILED to create view for ChatSession!" << std::endl;
    return NULL;
  }

  if(! _joined)
    join();

  view->makeVisible();
  return view;
}

void
SilcContact::slotCloseSession(void)
{
  QObject::disconnect(_chatSession, 0, this, 0);

  leave();
  _chatSession = NULL;
}

void
SilcContact::join(void)
{
  _joined = true;
}

void
SilcContact::leave(void)
{
  _joined = false;
}

void 
SilcContact::deleteContact(void)
{
  // we must not delete the contact, since it is referenced from within
  // the silc client library
  metaContact()->removeContact(this);
  
  Kopete::MetaContact *m = new Kopete::MetaContact();
  m->setTemporary(true);
  setMetaContact(m);
}

void
SilcContact::prettyPrintMessage(Kopete::Message &msg, int flags,
				SignatureStatus sigstat)
{
  if(!(flags & SILC_MESSAGE_FLAG_SIGNED)) 
    return;

  if(msg.direction() == Kopete::Message::Inbound
     || msg.direction() == Kopete::Message::Internal) {
    QString sigstatComment;

    switch(sigstat) {
    case Unknown:
      sigstatComment = i18n("unknown signature status");
      break;

    case Valid:
      sigstatComment = i18n("untrusted signature");
      break;

    case Trusted:
      sigstatComment = i18n("fully trusted signature.");
      break;

    case Failed:
      sigstatComment = i18n("verification FAILED.");
      break;
    }

    msg.setBody(QString::fromLatin1
		("<table width=\"100%\" border=0 cellspacing=0 cellpadding=0>"
		 "<tr><td class=\"highlight\"><font size=\"-1\">")
		+ i18n("Incoming Signed") + " "
		+ (flags & SILC_MESSAGE_FLAG_NOTICE
		   ? i18n("Notice") : (flags & SILC_MESSAGE_FLAG_ACTION
				       ? i18n("Action") : i18n("Message")))
		+ ": <b>" + sigstatComment 
		+ QString::fromLatin1("</b></font></td></tr>"
				      "<tr><td class=\"highlight\">")
		+ msg.escapedBody()
		+ QString::fromLatin1(" </td></tr></table>")
		, Kopete::Message::RichText );
  }

  else {
    msg.setBody(QString::fromLatin1
		("<table width=\"100%\" border=0 cellspacing=0 cellpadding=0>"
		 "<tr><td class=\"highlight\"><font size=\"-1\"><b>")
		+ i18n("Outgoing Signed") + " "
		+ (flags & SILC_MESSAGE_FLAG_ACTION
		   ? i18n("Action") : i18n("Message"))
		+ QString::fromLatin1(": </b></font></td></tr>"
				      "<tr><td class=\"highlight\">")
		+ msg.escapedBody()
		+ QString::fromLatin1(" </td></tr></table>")
		, Kopete::Message::RichText );
  }
}

SilcTK::SilcMime
SilcContact::mime_asm(SilcTK::SilcMime part)
{
  if(!part) return NULL;

  QString type = SilcTK::silc_mime_get_field(part, "Content-Type");
  if(type.isEmpty()) return NULL;

  if (SilcTK::silc_mime_is_partial(part)) {
    if (!mimeasm)  
      mimeasm = SilcTK::silc_mime_assembler_alloc();
    SilcTK::SilcMime mime = SilcTK::silc_mime_assemble(mimeasm, part);
    if(mime)
      return mime;
    return NULL;
  }
  return part;
}

QStringList *
SilcContact::saveMime(SilcTK::SilcMime mime)
{
  QStringList *list = new QStringList;
  if (SilcTK::silc_mime_is_multipart(mime)) {
    SilcTK::SilcMime p;
    const char *mixed;
    SilcTK::SilcDList parts = SilcTK::silc_mime_get_multiparts(mime, &mixed);

    /* Only "mixed" type supported */
    if (strcmp(mixed, "mixed"))
      return list;

    SilcTK::silc_dlist_start(parts);
    while ((p = (SilcTK::SilcMime)SilcTK::silc_dlist_get(parts))
	   != SILC_LIST_END) { 
      QStringList *tmp = saveMime(p);
      *list += *tmp;
    }
  }
  else {
    SilcTK::SilcUInt32 len;
    char temp[] = "/tmp/kopete.mime.XXXXXX";

    const char *data = (const char *)SilcTK::silc_mime_get_data(mime, &len);
    int fd = mkstemp(temp);
    if (!data || !len || fd == -1) return list;
    close(fd);

    QFile file(temp);
    file.open(IO_WriteOnly);
    file.writeBlock(data, len);
    file.close();

    list->append(file.name());
  }
  return list;
}


SilcTK::SilcMime
SilcContact::getMessageAsMime(Kopete::Message &msg)
{
  // prepare text/plain part
  SilcTK::SilcMime plain = SilcTK::silc_mime_alloc();
  SilcTK::silc_mime_add_field(plain, "MIME-Version", "1.0");
  SilcTK::silc_mime_add_field(plain, "Content-Transfer-Encoding", "binary");
  SilcTK::silc_mime_add_field(plain, "Content-Type",
			      "text/plain; charset=utf-8");

  QCString plaintext = msg.plainBody().utf8();
  SilcTK::silc_mime_add_data(plain, (unsigned char *) (const char *) plaintext,
			     plaintext.length());

  // prepare text/html part
  SilcTK::SilcMime html = SilcTK::silc_mime_alloc();
  SilcTK::silc_mime_add_field(html, "MIME-Version", "1.0");
  SilcTK::silc_mime_add_field(html, "Content-Transfer-Encoding", "binary");
  SilcTK::silc_mime_add_field(html, "Content-Type",
			      "text/html; charset=utf-8");

  QCString htmltext = msg.escapedBody().utf8();
  SilcTK::silc_mime_add_data(html, (unsigned char *) (const char *) htmltext,
			     htmltext.length());

  // no produce the multipart/alternative thingy
  SilcTK::SilcMime mime = SilcTK::silc_mime_alloc();
  SilcTK::silc_mime_set_multipart(mime, "alternative", "boundary");
  SilcTK::silc_mime_add_multipart(mime, plain);
  SilcTK::silc_mime_add_multipart(mime, html);

  return mime;
}


SilcTK::SilcDList 
SilcContact::getFileAsMime(const QString &fileName) 
{
  QFile file(fileName);
  SilcTK::SilcMime mime = SilcTK::silc_mime_alloc();

  SilcTK::silc_mime_add_field(mime, "MIME-Version", "1.0");
  SilcTK::silc_mime_add_field(mime, "Content-Transfer-Encoding", "binary");
  
  KMimeMagicResult *mtype = KMimeMagic::self()->findFileType(fileName);
  if(mtype && mtype->isValid())
    SilcTK::silc_mime_add_field(mime, "Content-Type", 
                                mtype->mimeType().latin1());
  else
    SilcTK::silc_mime_add_field(mime, "Content-Type", 
                                "application/octet-stream");
  file.open(IO_ReadOnly);
  QByteArray content = file.readAll();
  SilcTK::silc_mime_add_data(mime, (unsigned char *)content.data(), content.size());

  SilcTK::SilcDList ret = SilcTK::silc_mime_encode_partial(mime, 23 << 10);
  SilcTK::silc_mime_free(mime);

  return ret;
}

void
SilcContact::sendFileAsMime(const QString &) {}


QString
SilcContact::mimeDisplayMessage(QString fname, int outgoing_chunks)
{
  QFileInfo info(fname);

  /* find mime type */
  KMimeMagicResult *mtype = KMimeMagic::self()->findFileType(fname);
  QString ctype = (mtype && mtype->isValid()) 
    ? QString(mtype->mimeType()) : QString(i18n("unknown"));
    

  QString msg =
    QString(outgoing_chunks
	    ? i18n("<a href=\"%2\">MIME message</a> (%1)")
	    : i18n("New MIME message (%1) saved <a href=\"%2\">here</a>"))
    .arg(ctype).arg(fname);

  if(outgoing_chunks)
    msg.append(QString(" sent in %1 chunks").arg(outgoing_chunks));

  msg.append(QString(" (%1.%2 Kbyte)")
	     .arg((info.size() / 1024))
	     .arg((info.size() % 1024) / 100));

  if(account()->displayImagesInline()
     && ctype.left(6).compare("image/") == 0) {
    QImage img(fname);
    int h = img.height();
    int w = img.width();

    if(w > 400) {
      h = h * 400 / w;
      w = 400;
    }

    if(h > 300) {
      w = w * 300 / h;
      h = 300;
    }

    msg.append(QString("<br><img src=\"%1\" width=\"%2\" height=\"%3\">")
	       .arg(fname).arg(w).arg(h));
  }

  return msg;
}
#include "silccontact.moc"
