//                       -*- mode: C++ -*-
// Copyright(C) 2005,2007 Stefan Siegl <stesie@brokenpipe.de>
// Copyright(C) 2007 Christian Dietrich <stettberger@brokenpipe.de>
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef KOPETESILC_SILCCONTACT_H
#define KOPETESILC_SILCCONTACT_H

#include <kopetecontact.h>

namespace Kopete {
  class ChatSession;
  class MetaContact;
  class Message;
}

// include libsilc stuff into a separate namespace
namespace SilcTK {
#include <silc.h>
#include <silcclient.h>
}

class KopeteView;

class SilcAccount;
class SilcChatSession;

/**
 * @brief abstract reimplementation of Kopete::Contact class
 *
 * Every channel and every user of the connected SILC network will have it's
 * very own representation in kopete_silc.  SilcContact is an abstract
 * parent class of SilcChannelContact (representing channels) and
 * SilcBuddyContact (specialized for buddies, i.e. users)
 *
 * @author Stefan Siegl <stesie@brokenpipe.de>
 */
class SilcContact : public Kopete::Contact {
  Q_OBJECT;

public:
  enum SignatureStatus {
    /**
     * the signer is not known or the signature couldn't be verified for
     * whatever reason.
     */
    Unknown,

    /** 
     * the signature could be verified successfully
     */
    Valid,

    /** 
     * the signature could be verified and the public key is marked as trusted
     */
    Trusted,

    /**
     * failed, the signature is invalid
     */
    Failed
  };



  SilcContact(SilcAccount *account, const QString &nickname, 
	      Kopete::MetaContact *, const QString &icon = QString::null);
  virtual ~SilcContact();

  /**
   * @brief return the primary message manager of this contact
   */
  virtual Kopete::ChatSession *manager(Kopete::Contact::CanCreateFlags =
				       Kopete::Contact::CannotCreate);

  /**
   * @brief return KopeteView for this channel (or create a new one)
   *
   * This is somewhat a convenience function, call it to force a chat window
   * being opened for this channel or one-on-one chat.
   */
  virtual KopeteView *view(void);

  /**
   * @brief join the SilcContact
   *
   * Default implementation does nothing.
   */
  virtual void join(void);

  /**
   * @brief leave the channel (if joined)
   *
   * Default implementation does nothing.
   */
  virtual void leave(void);

  /**
   * @brief access this contact's Kopete::Account subclass (aka SilcAccount)
   */
  SilcAccount *account(void); 

  /**
   * @brief signals in the chat window that a message has been signed
   */
  static void prettyPrintMessage(Kopete::Message &msg, int flags,
				 SignatureStatus status = Unknown);
  /**
   * @brief assembles mime parts, if mime message is complete return it
   */
  SilcTK::SilcMime mime_asm(SilcTK::SilcMime part);
 
  /**
   * @brief save mime message and return a list of files where parts were
   * saved
   */
  QStringList *saveMime(SilcTK::SilcMime mime);

  /**
   * @brief send file as mime to contact
   */
  virtual void sendFileAsMime(const QString &fileName);

  /**
   * @brief generate a message to display for incoming SilcMime message
   */
  virtual QString mimeDisplayMessage(QString fname, int out_chunks = 0);

  /**
   * @brief creates an mime message form file
   */
  SilcTK::SilcDList getFileAsMime(const QString &fileName);

  /**
   * @brief produce a multipart/alternative SilcMime from a Kopete::Message
   */
  SilcTK::SilcMime getMessageAsMime(Kopete::Message &msg);


protected:
  /** 
   * @brief the Kopete::ChatSession associated to this SilcContact
   */
  SilcChatSession *_chatSession;

  /**
   * @brief whether or not the channel is currently joined
   */
  bool _joined;

public slots:
  virtual void deleteContact(void);
  
private slots:
  /**
   * @brief send outgoing message
   */
  virtual void slotSendMessage(Kopete::Message &, Kopete::ChatSession *) = 0;

  /**
   * @brief the Kopete::ChatSession was closed
   */
  void slotCloseSession(void);
private:
  /**
   * @brief silcmimeassembler
   */
  SilcTK::SilcMimeAssembler mimeasm;
};

#endif // KOPETESILC_SILCCONTACT_H
