/*
 *  config.c
 *  mod_musicindex
 *
 *  $Id: config.c 662 2006-07-09 21:05:07Z varenet $
 *
 *  Created by Thibaut VARENE on Thu Mar 20 2003.
 *  Copyright (c) 2003-2006 Regis BOUDIN
 *  Copyright (c) 2003-2005 Thibaut VARENE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version 2.1,
 *  as published by the Free Software Foundation.
 *
 */

/**
 * @file
 * Configuration handling and default settings.
 *
 * @author Regis Boudin
 * @version $Revision: 662 $
 * @date 2003-2005
 *
 * That file takes care of the module configuration, either via setting some
 * known default values, and/or by getting them from environmental configuration
 * variables passed to apache in its config file.
 *
 * @todo Regis, please document.
 */

#include "config.h"
#include "sort.h"

#include "cache-file.h"

#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>	/* atoi */
#endif

typedef int (*cache_backend_setup)(apr_pool_t *pool, const char *setup_string, mu_config *const conf);

static cache_backend_setup cache_setups[] = {
#ifdef	ENABLE_CACHE_FILE
	cache_file_setup,
#endif
	NULL
};

static const char default_sort[] = "APNBTLREFMU";	/**< Default sort order */
static const char default_fields[] = "TBLR";		/**< Default fields displayed */

static const char default_directory[] = "/musicindex";	/**< Default musicindex directory */
static const char default_rootname[] = "Music";		/**< Default root directory name */
static const char default_css[] = "musicindex.css";	/**< Default CSS file name */

/**
 * Generates a per-directory configuration.
 *
 * This function creates a new configuration structure and fills it with
 * default values.
 *
 * @param p Apache pool for memory allocation.
 * @param dummy String given by apache we don't use.
 *
 * @return The newly created configuration structure.
 */
void *create_musicindex_config(apr_pool_t *p, char *dummy)
{
	register unsigned short i;
	mu_config *const new = (mu_config *)ap_pcalloc(p, sizeof(mu_config));

	for (i = 0; i < ARG_NUMBER; i++) {
		new->order[i] = SB_DEFAULT;
		new->fields[i] = '\0';
	}

	strcpy(new->fields, default_fields);
	strcpy(new->order, default_sort);

	new->title = default_rootname;

	new->directory = default_directory;
	new->css = default_css;
	new->search = NULL;
	new->iceserver = NULL;

	new->cache = NULL;
	new->cache_setup = NULL;

	new->cookie_life = CONF_COOKIE_LIFE;
	new->rss_items = 0;

	new->options = 0;
	new->options_not = 0;

	new->dir_per_line = CONF_DIRPERLINE;

	set_fctptrs(new);

	return (void *)new;
}

/**
 * Merge per-directory configurations.
 *
 * This function is supposed to merge two per-directory configurations. We
 * simply copy it from the parent directory.
 *
 * @param p Apache pool for memory allocation.
 * @param basev Pointer to main configuration structure.
 * @param addv Pointer to parent dir configuration structure,
 * 	is *apparently* equal to basev if no specific dir conf was set.
 *
 * @return The newly created configuration structure.
 */
void *merge_musicindex_configs(apr_pool_t *p, void *basev, void *addv)
{
	mu_config *const new = (mu_config *)ap_pcalloc(p, sizeof(mu_config));
	mu_config *const base = (mu_config *)basev;
	mu_config *const add = (mu_config *)addv;

	if (strcmp(add->order, default_sort))
		strcpy(new->order, add->order);
	else
		strcpy(new->order, base->order);

	if (strcmp(add->fields, default_fields))
		strcpy(new->fields, add->fields);
	else
		strcpy(new->fields, base->fields);

	if (add->title != default_rootname)
		new->title = ap_pstrdup(p, add->title);
	else if (base->title != default_rootname)
		new->title = ap_pstrdup(p, base->title);
	else
		new->title = default_rootname;

	if (add->directory != default_directory)
		new->directory = ap_pstrdup(p, add->directory);
	else if (base->directory != default_directory)
		new->directory = ap_pstrdup(p, base->directory);
	else
		new->directory = default_directory;

	if (add->css != default_css)
		new->css = ap_pstrdup(p, add->css);
	else if (base->css != default_css)
		new->css = ap_pstrdup(p, base->css);
	else
		new->css = default_css;

	if ((add->cache != NULL) && (add->cache_setup != NULL)) {
		new->cache = add->cache;
		new->cache_setup = add->cache_setup;
	} else if (base->cache != NULL) {
		new->cache = base->cache;
		new->cache_setup = base->cache_setup;
	} else {
		new->cache = NULL;
		new->cache_setup = NULL;
	}

	if (add->iceserver) /* new server defined ? */
		new->iceserver = ap_pstrdup(p, add->iceserver);
	else if (base->iceserver) /* previous sever defined ? */
		new->iceserver = ap_pstrdup(p, base->iceserver);
	else /* OK, no server defined... */
		new->iceserver = NULL;

	new->options = base->options | add->options;
	new->options &= ~(add->options_not);

	new->options_not = add->options_not;

	if (add->cookie_life != CONF_COOKIE_LIFE)
		new->cookie_life = add->cookie_life;
	else
		new->cookie_life = base->cookie_life;

	if (add->rss_items != 0)
		new->rss_items = add->rss_items;
	else
		new->rss_items = base->rss_items;

	if (add->dir_per_line != CONF_DIRPERLINE)
		new->dir_per_line = add->dir_per_line;
	else
		new->dir_per_line = base->dir_per_line;

	set_fctptrs(new);

	return new;
}

static const struct {
	const char *string;
	const char value;
} options[] = {
	{"track",	SB_TRACK},
	{"disc",	SB_POSN},
	{"length",	SB_LENGTH},
	{"bitrate",	SB_BITRATE},
	{"freq",	SB_FREQ},
	{"artist",	SB_ARTIST},
	{"album",	SB_ALBUM},
	{"title",	SB_TITLE},
	{"filename",	SB_FILENAME},
	{"date",	SB_DATE},
	{"filetype",	SB_FILETYPE},
	{"genre",	SB_GENRE},
	{"uri",		SB_URI},
	{"size",	SB_SIZE},
	{NULL, 		'\0'}
};

void sort_or_fields(cmd_parms *cmd, char *list, const char *optstr)
{
	const char *r;
	register unsigned short i = 0, j;

	while (optstr[0] && (i < ARG_NUMBER)) {
		r = ap_getword_conf(cmd->pool, &optstr);
		for (j=0; options[j].string; j++) {
			if (!strcasecmp(r, options[j].string)) {
				list[i++] = options[j].value;
				break;
			}
		}
	}
	if (i == ARG_NUMBER)
		i--;
	list[i] = '\0';
}

static const char *sort_order(cmd_parms *cmd, void *d, const char *optstr)
{
	mu_config *const d_cfg = (mu_config *)d;
	sort_or_fields(cmd, d_cfg->order, optstr);
	set_fctptrs(d_cfg);
	return NULL;
}

static const char *set_fields(cmd_parms *cmd, void *d, const char *optstr)
{
	sort_or_fields(cmd, ((mu_config *)d)->fields, optstr);
	return NULL;
}

/**
 * Enables the module options.
 *
 * Read the different options given as parameters, and set the various flags.
 *
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *basic_config(cmd_parms *cmd, void *d, const char *optstr)
{
	const char *r;
	mu_config *const cfg = (mu_config *)d;
	
	while (optstr[0] != '\0') {
		int enable = 1;
		unsigned short flag = 0;
		r = ap_getword_conf(cmd->pool, &optstr);

		/* Should we enable or disable ? Is there a +/- prefix ? */
		if ( *r == '-' ) {
			enable = 0;
			r++;
		} else if (*r == '+') {
			r++;
		}

		/* Which option are we talking about ? */
		if (strcmp(r, "On") == 0) {
			enable = 1;
			flag = MI_ACTIVE;
		} else if (strcmp(r, "Off") == 0) {
			enable = 0;
			flag = MI_ACTIVE;
		} else if (strcmp(r, "Stream") == 0) {
			flag = MI_ALLOWSTREAM;
		} else if (strcmp(r, "Download") == 0) {
			flag = MI_ALLOWDWNLD;
		} else if (strcmp(r, "Search") == 0) {
			flag = MI_ALLOWSEARCH;
#ifdef ENABLE_OUTPUT_ARCHIVE
		} else if (strcmp(r, "Tarball") == 0) {
			flag = MI_ALLOWTARBALL;
#endif
		} else if (strcmp(r, "Rss") == 0) {
			if (enable == 1)
				cfg->rss_items = CONF_RSS_ITEMS;
			else {
				cfg->rss_items = -1;
			}
		}

		/* Have this done in one place instead of five */
		if (flag != 0) {
			if (enable == 1) {
				cfg->options |= flag;
				cfg->options_not &= ~flag;
			} else {
				cfg->options &= ~flag;
				cfg->options_not |= flag;
			}
		}
	}

	return NULL;
}

/**
 * Configures the cache location in the filesystem.
 *
 * This function sets the configuration string for the cache subsystem.
 *
 * @param cmd Struct containing a pointer to the pool I have to use.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *set_cache_uri(cmd_parms *cmd, void *d, const char *optstr)
{
	int i, ret = 1;

	for (i = 0; (ret != 0) && (cache_setups[i] != NULL); i++)
		ret = cache_setups[i](cmd->pool, optstr, (mu_config *)d);

	return NULL;
}

/**
 * Sets the title of the page.
 *
 * This function simply chages the name of the root.
 *
 * @param cmd Struct containing a pointer to the pool I have to use.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *set_page_title(cmd_parms *cmd, void *d, const char *optstr)
{
	mu_config *const cfg = (mu_config *)d;
	if ((optstr == NULL) || (optstr[0] == '\0'))
#ifdef NEW_HEADER
		cfg->title = NULL;
#else
		cfg->title = ap_pstrdup(cmd->pool, default_rootname);
#endif
	else
		cfg->title = ap_pstrdup(cmd->pool, optstr);
	return NULL;
}

/**
 * Sets the icecast server address.
 * This function sets an icecast server for streaming.
 *
 * @param cmd Struct containing a pointer to the pool I have to use.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *set_ice_server(cmd_parms *cmd, void *d, const char *optstr)
{
	mu_config *const cfg = (mu_config *)d;
	cfg->iceserver = ap_pstrdup(cmd->pool, optstr);
	return NULL;
}

/**
 * Sets the default CSS file.
 * This function sets which CSS file will be used by default
 *
 * @param cmd Struct containing a pointer to the pool I have to use.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *set_css_default(cmd_parms *cmd, void *d, const char *optstr)
{
	((mu_config *)d)->css = ap_pstrdup(cmd->pool, optstr);
	return NULL;
}

/**
 * Sets the cookie lifetime.
 * This function sets the lifetime in seconds of the cookie used for custom
 * playlist constructs.
 *
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *set_cookie_life(cmd_parms *cmd, void *d, const char *optstr)
{
	((mu_config *)d)->cookie_life = atoi(optstr);
	return NULL;
}

/**
 * Sets the number of directories to display per line.
 *
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *set_dirperline(cmd_parms *cmd, void *d, const char *optstr)
{
	((mu_config *)d)->dir_per_line = atoi(optstr);
	return NULL;
}

/**
 * Sets the default display.
 * This function sets an icecast server for streaming.
 *
 * @param cmd Struct containing a pointer to the pool I have to use.
 * @param d Pointer to configuration structure.
 * @param optstr The string given as parameter in the configuration.
 *
 * @return NULL don't ask why.
 */
static const char *set_display(cmd_parms *cmd, void *d, const char *optstr)
{
	mu_config *const cfg = (mu_config *)d;
	if (strcmp(optstr, "RSS") == 0) {
		cfg->options |= MI_RSS;
		cfg->order[0] = SB_MTIME;
		cfg->order[1] = SB_URI;
		cfg->options &= ~MI_RECURSIVE;
		cfg->rss_items = CONF_RSS_ITEMS;
	}
	else if (strcmp(optstr, "HTML") == 0) {
		cfg->options &= ~MI_RSS;
		cfg->options_not |= MI_RSS;
		strcpy(cfg->order, default_sort);
	}
	return NULL;
}

/************************* Deprecated options *******************************/

/**
 * Enables the module.
 *
 * This function simply sets the module as active.
 *
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param yes Boolean flag setup by apache if the configuration option is enabled.
 *
 * @return NULL don't ask why.
 */
static const char *music_lister(cmd_parms *cmd, void *d, int yes)
{
	mu_config *const cfg = (mu_config *)d;
	if (yes)
		cfg->options |= MI_ACTIVE;
	else {
		cfg->options &= ~MI_ACTIVE;
		cfg->options_not |= MI_ACTIVE;
	}

	return NULL;
}

/**
 * Enables the stream in the module.
 *
 * This function simply sets the streaming function of the module as active.
 *
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param yes Boolean flag setup by apache if the configuration option is enabled.
 *
 * @return NULL don't ask why.
 */
static const char *allow_stream(cmd_parms *cmd, void *d, int yes)
{
	mu_config *const cfg = (mu_config *)d;
	if (yes)
		cfg->options |= MI_ALLOWSTREAM;
	else {
		cfg->options &= ~MI_ALLOWSTREAM;
		cfg->options_not |= MI_ALLOWSTREAM;
	}

	return NULL;
}

/**
 * Enables the download in the module.
 *
 * This function simply sets the downloading function of the module as active.
 *
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param yes Boolean flag setup by apache if the configuration option is enabled.
 *
 * @return NULL don't ask why.
 */
static const char *allow_download(cmd_parms *cmd, void *d, int yes)
{
	mu_config *const cfg = (mu_config *)d;
	if (yes)
		cfg->options |= MI_ALLOWDWNLD;
	else {
		cfg->options &= ~MI_ALLOWDWNLD;
		cfg->options_not |= MI_ALLOWDWNLD;
	}

	return NULL;
}

/**
 * Enables the search in the module.
 *
 * This function simply sets the searching function of the module as active.
 *
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param yes Boolean flag setup by apache if the configuration option is enabled.
 *
 * @return NULL don't ask why.
 */
static const char *allow_search(cmd_parms *cmd, void *d, int yes)
{
	mu_config *const cfg = (mu_config *)d;
	if (yes)
		cfg->options |= MI_ALLOWSEARCH;
	else {
		cfg->options &= ~MI_ALLOWSEARCH;
		cfg->options_not |= MI_ALLOWSEARCH;
	}

	return NULL;
}

/**
 * Sets the RSS feed number of items.
 * This function sets the max number of new items available in a RSS feed.
 *
 * @param cmd I don't use it.
 * @param d Pointer to configuration structure.
 * @param yes Boolean flag setup by apache if the configuration option is enabled.
 *
 * @return NULL don't ask why.
 */
static const char *allow_rss(cmd_parms *cmd, void *d, int yes)
{
	mu_config *const cfg = (mu_config *)d;
	if (yes)
		cfg->rss_items = CONF_RSS_ITEMS;
	else {
		cfg->rss_items = -1;
	}
	return NULL;
}

/********************* This bit is NOT deprecated, keep it ! *****************/
/* XXX if somebody puts a ACCESS_CONF|RSRC_CONF setting in .htaccess, the module
seems to go totally foobar */
const command_rec musicindex_cmds[] = {
	AP_INIT_RAW_ARGS("MusicIndex", basic_config, NULL, OR_INDEXES,
		"can be : On/Off +/-Stream +/-Download +/-Search +/-Rss"),
	AP_INIT_RAW_ARGS("MusicSortOrder", sort_order, NULL, OR_INDEXES,
		"can be : title album artist track disc length bitrate filetype genre filename date uri"),
	AP_INIT_RAW_ARGS("MusicFields", set_fields, NULL, OR_INDEXES,
		"can be : title album artist track disc length bitrate filetype genre filename date"),
	AP_INIT_RAW_ARGS("MusicIndexCache", set_cache_uri, NULL, ACCESS_CONF|RSRC_CONF,
		"Set the cache configuration string"),
	AP_INIT_RAW_ARGS("MusicPageTitle", set_page_title, NULL, OR_INDEXES,
		"Set the root title of the page."),
	AP_INIT_TAKE1("MusicIceServer", set_ice_server, NULL, ACCESS_CONF|RSRC_CONF,
		"Set the icecast server address : [server.domain.org]:8000"),
	AP_INIT_TAKE1("MusicDefaultCss", set_css_default, NULL, OR_INDEXES,
		"Set the default CSS file"),
	AP_INIT_TAKE1("MusicCookieLife", set_cookie_life, NULL, ACCESS_CONF|RSRC_CONF,
		"Set the lifetime (in seconds) of the cookie sent for custom playlists"),
	AP_INIT_TAKE1("MusicDefaultDisplay", set_display, NULL, ACCESS_CONF|RSRC_CONF,
		"Set the default display returned by the module (HTML or RSS)"),
	AP_INIT_TAKE1("MusicDirPerLine", set_dirperline, NULL, OR_INDEXES,
		"Set the number of directories per line in the directory listing"),
	/* Deprecated soon */
	AP_INIT_FLAG("MusicAllowStream", allow_stream, NULL, OR_INDEXES,
		"Deprecated (Allow streaming)"),
	AP_INIT_FLAG("MusicAllowDownload", allow_download, NULL, OR_INDEXES,
		"Deprecated (Allow downloading)"),
	AP_INIT_FLAG("MusicAllowSearch", allow_search, NULL, OR_INDEXES,
		"Deprecated (Enable searching) (EXPERIMENTAL)"),
	AP_INIT_FLAG("MusicLister", music_lister, NULL, OR_INDEXES,
		"Deprecated (Enable the Musicindex index maker)"),
	AP_INIT_FLAG("MusicAllowRss", allow_rss, NULL, OR_INDEXES,
		"Deprecated (Allow RSS feed)"),
	{NULL}
};

