/**
 * ===========================================
 * LibBase : a free Java utility library
 * ===========================================
 *
 * Project Info:  http://reporting.pentaho.org/libbase
 *
 * (C) Copyright 2007,2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 * ------------
 * PropertyFileConfiguration
 * ------------
 * (C) Copyright 2006, by Pentaho Corporation.
 */


package org.pentaho.reporting.libraries.base.config;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.pentaho.reporting.libraries.base.util.ObjectUtilities;

/**
 * A report configuration that reads its values from an arbitary property file.
 *
 * @author Thomas Morgner
 */
public class PropertyFileConfiguration extends HierarchicalConfiguration
{
  private static final Log LOGGER = LogFactory.getLog(PropertyFileConfiguration.class);
  private static final long serialVersionUID = 2423181637547944866L;

  /**
   * Default constructor.
   */
  public PropertyFileConfiguration()
  {
    // nothing required
  }

  /**
   * Lods the property file from a classpath resource name. The classpath resource must be loadable via
   * <code>PropertyFileConfiguration.class.getResource(..)</code>
   * 
   * @param resourceName the resource name to be loaded.
   */
  public void load(final String resourceName)
  {
    load(resourceName, PropertyFileConfiguration.class);
  }

  /**
   * Loads the properties stored in the given file. This method does nothing if
   * the file does not exist or is unreadable. Appends the contents of the loaded
   * properties to the already stored contents.
   *
   * @param resourceName the file name of the stored properties.
   * @param resourceSource the class to which relative resource paths are resolved.
   */
  public void load(final String resourceName, final Class resourceSource)
  {
    final InputStream in = ObjectUtilities.getResourceRelativeAsStream
        (resourceName, resourceSource);
    if (in != null)
    {
      try
      {
        load(in);
      }
      finally
      {
        try
        {
          in.close();
        }
        catch (IOException e)
        {
          // ignore
        }
      }
    }
    else
    {
      LOGGER.debug("Configuration file not found in the classpath: " + resourceName);
    }

  }

  /**
   * Loads the properties stored in the given file. This method does nothing if
   * the file does not exist or is unreadable. Appends the contents of the loaded
   * properties to the already stored contents.
   *
   * @param in the input stream used to read the properties.
   */
  public void load(final InputStream in)
  {
    if (in == null)
    {
      throw new NullPointerException();
    }

    try
    {
      final BufferedInputStream bin = new BufferedInputStream(in);
      final Properties p = new Properties();
      p.load(bin);
      this.getConfiguration().putAll(p);
      bin.close();
    }
    catch (IOException ioe)
    {
      LOGGER.warn("Unable to read configuration", ioe);
    }

  }

}