//
// ToolStripMenuItemTest.cs
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Copyright (c) 2006 Jonathan Pobst
//
// Authors:
//	Jonathan Pobst (monkey@jpobst.com)
//
#if NET_2_0
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using System.Drawing;
using System.Windows.Forms;
using System.ComponentModel;

namespace MonoTests.System.Windows.Forms
{
	[TestFixture]
	public class ToolStripMenuItemTest
	{
		[Test]
		public void Constructor ()
		{
			ToolStripMenuItem tsi = new ToolStripMenuItem ();

			Assert.AreEqual (false, tsi.Checked, "A1");
			Assert.AreEqual (false, tsi.CheckOnClick, "A2");
			Assert.AreEqual (CheckState.Unchecked, tsi.CheckState, "A3");
			Assert.AreEqual (true, tsi.Enabled, "A4");
			Assert.AreEqual (false, tsi.IsMdiWindowListEntry, "A5");
			Assert.AreEqual (ToolStripItemOverflow.Never, tsi.Overflow, "A6");
			Assert.AreEqual (null, tsi.ShortcutKeyDisplayString, "A7");
			Assert.AreEqual (Keys.None, tsi.ShortcutKeys, "A8");
			Assert.AreEqual (true, tsi.ShowShortcutKeys, "A9");
			Assert.AreEqual (SystemColors.ControlText, tsi.ForeColor, "A9-1");
			
			Assert.AreEqual ("System.Windows.Forms.ToolStripMenuItem+ToolStripMenuItemAccessibleObject", tsi.AccessibilityObject.GetType ().ToString (), "A10");
			int count = 0;
			EventHandler oc = new EventHandler (delegate (object sender, EventArgs e) { count++; });
			Image i = new Bitmap (1, 1);

			tsi = new ToolStripMenuItem (i);
			tsi.PerformClick ();
			Assert.AreEqual (null, tsi.Text, "A10-1");
			Assert.AreSame (i, tsi.Image, "A10-2");
			Assert.AreEqual (0, count, "A10-3");
			Assert.AreEqual (string.Empty, tsi.Name, "A10-4");

			tsi = new ToolStripMenuItem ("Text");
			tsi.PerformClick ();
			Assert.AreEqual ("Text", tsi.Text, "A10-5");
			Assert.AreSame (null, tsi.Image, "A11");
			Assert.AreEqual (0, count, "A12");
			Assert.AreEqual (string.Empty, tsi.Name, "A13");

			tsi = new ToolStripMenuItem ("Text", i);
			tsi.PerformClick ();
			Assert.AreEqual ("Text", tsi.Text, "A14");
			Assert.AreSame (i, tsi.Image, "A15");
			Assert.AreEqual (0, count, "A16");
			Assert.AreEqual (string.Empty, tsi.Name, "A17");

			tsi = new ToolStripMenuItem ("Text", i, oc);
			tsi.PerformClick ();
			Assert.AreEqual ("Text", tsi.Text, "A18");
			Assert.AreSame (i, tsi.Image, "A19");
			Assert.AreEqual (1, count, "A20");
			Assert.AreEqual (string.Empty, tsi.Name, "A21");

			tsi = new ToolStripMenuItem ("Text", i, oc, "Name");
			tsi.PerformClick ();
			Assert.AreEqual ("Text", tsi.Text, "A22");
			Assert.AreSame (i, tsi.Image, "A23");
			Assert.AreEqual (2, count, "A24");
			Assert.AreEqual ("Name", tsi.Name, "A25");
		}

		[Test]
		public void BehaviorKeyboardShortcuts ()
		{
			ExposeProtectedMethods tsmi = new ExposeProtectedMethods ();
			tsmi.ShortcutKeys = Keys.Control | Keys.D;

			Message m = new Message ();
			Assert.AreEqual (false, tsmi.PublicProcessCmdKey (ref m, Keys.D), "A1");
			Assert.AreEqual (false, tsmi.PublicProcessCmdKey (ref m, Keys.Control), "A2");
			Assert.AreEqual (true, tsmi.PublicProcessCmdKey (ref m, Keys.Control | Keys.D), "A3");
			Assert.AreEqual (false, tsmi.PublicProcessCmdKey (ref m, Keys.A), "A4");
			Assert.AreEqual (false, tsmi.PublicProcessCmdKey (ref m, Keys.Control | Keys.A), "A5");
			
			tsmi.ShowShortcutKeys = false;
			Assert.AreEqual (true, tsmi.PublicProcessCmdKey (ref m, Keys.Control | Keys.D), "A6");
			
			tsmi.ShortcutKeyDisplayString = "Moose";
			Assert.AreEqual (true, tsmi.PublicProcessCmdKey (ref m, Keys.Control | Keys.D), "A7");
		}
		
		[Test]
		public void BehaviorMdiWindowMenuItem ()
		{
			Form f = new Form ();
			f.ShowInTaskbar = false;
			f.IsMdiContainer = true;
			Form c1 = new Form ();
			c1.MdiParent = f;
			Form c2 = new Form ();
			c2.MdiParent = f;				
		
			MenuStrip ms = new MenuStrip ();
			ToolStripMenuItem tsmi = (ToolStripMenuItem)ms.Items.Add ("Window");
			f.Controls.Add (ms);
			ms.MdiWindowListItem = tsmi;
			
			f.MainMenuStrip = ms;

			c1.Show ();
			f.Show ();
			Assert.AreEqual (true, (tsmi.DropDownItems[0] as ToolStripMenuItem).IsMdiWindowListEntry, "R1");
			
			f.Close ();
		}
		
		[Test]
		public void BehaviorShortcutText ()
		{
			ToolStripMenuItem tsmi = new ToolStripMenuItem ();
			
			tsmi.ShortcutKeys = Keys.Control | Keys.O;
			
			Assert.AreEqual (null, tsmi.ShortcutKeyDisplayString, "A1");
			
			tsmi.ShortcutKeyDisplayString = "Test String";
			Assert.AreEqual ("Test String", tsmi.ShortcutKeyDisplayString, "A2");

			tsmi.ShortcutKeys = Keys.Control | Keys.P;
			Assert.AreEqual ("Test String", tsmi.ShortcutKeyDisplayString, "A3");

			tsmi.ShortcutKeyDisplayString = string.Empty;
			Assert.AreEqual (string.Empty, tsmi.ShortcutKeyDisplayString, "A4");

			tsmi.ShortcutKeyDisplayString = null;
			Assert.AreEqual (null, tsmi.ShortcutKeyDisplayString, "A5");
		}
		
		private class ExposeProtectedMethods : ToolStripMenuItem
		{
			public bool PublicProcessCmdKey (ref Message m, Keys keys)
			{
				return this.ProcessCmdKey (ref m, keys);
			}
		}
	}
}
#endif
