//
//  MWReverseEngineering.m
//  MySQL Workbench
//
//  Created by Alfredo Kojima on 05/8/30.
//  Copyright 2005 MySQL AB. All rights reserved.
//

#import "MWReverseEngineering.h"
#import <MySQLGRT/MGRTRevEngFilterPane.h>
#import <MySQLToolsCommon/myxutil.h>
#import <MySQLToolsCommon/mxUtils.h>
#import <MySQLToolsCommon/MVerticalBox.h>

@implementation MWReverseEngineering

- (id)initWithMGRT:(MGRT*)grt
{
  self= [super initWithWindowNibName:@"ReverseEngineering"];
  if (self)
  {
    _grt= [grt retain];
    [tabView selectFirstTabViewItem:nil];

    [self loadWindow];

    _taskUnchecked= [[NSImage imageNamed:@"task_unchecked"] retain];
    _taskChecked= [[NSImage imageNamed:@"task_checked"] retain];
    _taskError= [[NSImage imageNamed:@"task_error"] retain];
    _taskDisabled= [[NSImage imageNamed:@"task_disabled"] retain];
    
    _section= 0;
    _schemaListObjects= [[NSMutableArray alloc] init];
    _filterPanels= [[NSMutableArray alloc] init];
    
    MGRTValue migration(MGRTValue::createObject([_grt grt], "db.migration.Migration", "Migration"));
    [_grt setGlobalValue:migration.grtValue() forPath:"/migration"];
  }
  return self;
}



- (void)dealloc 
{
  [_taskUnchecked release];
  [_taskChecked release];
  [_taskError release];
  [_taskDisabled release];
  
  [_schemaListObjects release];
  [_filterPanels release];
  delete _catalog;
  delete _schemaList;
  
  [[NSNotificationCenter defaultCenter] removeObserver:self];
  [_grt release];
  [super dealloc];
}


- (void)awakeFromNib
{
  NSView *rdbmsV;
  NSView *paramsV;
  NSView *advParamsV;

  _connPanel= [[MGRTConnectionPanel alloc] initWithMGRT:_grt
                                        connectionsPath:@"/rdbmsMgmt"
                                             targetPath:@"/workbench/connection"];
  [_connPanel setDelegate:self];

  rdbmsV= [_connPanel rdbmsPanel];
  paramsV= [_connPanel paramsPanel];
  advParamsV= [_connPanel advParamsPanel]; 
  
  [rdbmsV retain];
  [rdbmsV removeFromSuperview];
  [rdbmsBox addSubview:rdbmsV];
  [rdbmsV release];

  [rdbmsV setFrame:NSMakeRect(16, 11, NSWidth([rdbmsBox frame])-32, NSHeight([rdbmsV frame]))];
  
  [paramsV retain];
  [paramsV removeFromSuperview];
  [paramBox addSubview:paramsV];
  [paramsV release];
  [paramsV setPostsFrameChangedNotifications:YES];
  
  [[NSNotificationCenter defaultCenter] addObserver:self
                                           selector:@selector(panelResized:)
                                               name:NSViewFrameDidChangeNotification
                                             object:paramsV];
    
  [paramsV setFrame:NSMakeRect(16, 11, NSWidth([paramBox frame])-32, NSHeight([paramBox frame])-35)];
  
  [advParamsV retain];
  [advParamsV removeFromSuperview];
  [advParamBox addSubview:advParamsV];
  [advParamsV release];
  [advParamsV setPostsFrameChangedNotifications:YES];
  
  [[NSNotificationCenter defaultCenter] addObserver:self
                                           selector:@selector(panelResized:)
                                               name:NSViewFrameDidChangeNotification
                                             object:advParamsV];
  
  [advParamsV setFrame:NSMakeRect(16, 11, NSWidth([advParamBox frame])-32, NSHeight([advParamBox frame])-35)];
  
  
  [_connPanel setSelectRdbms];
  // add the connect panel pieces
  [_connPanel refreshRdbmsInfo];  
  
  [progress setHidden:YES];
  
  [filterScrollView setHasVerticalScroller:YES];
  
  _filterContent= [[MVerticalBox alloc] initWithFrame:MXRectWithSize([filterScrollView contentSize])];
  [_filterContent setSpacing: 10];
  [filterScrollView setDocumentView:_filterContent];
  [_filterContent release];
}


- (void)connectionPanel:(MGRTConnectionPanel*)panel readinessChanged:(BOOL)flag
{
  if (_section == 0)
  {
    [nextButton setEnabled:flag];
  }
}


- (void)panelResized:(NSNotification*)notif
{
  if ([notif object] == [_connPanel paramsPanel])
  {
    [[_connPanel paramsPanel] setFrame:NSMakeRect(16, 11, NSWidth([paramBox frame])-32, NSHeight([paramBox frame])-35)];
  }
  else
  {
    [[_connPanel advParamsPanel] setFrame:NSMakeRect(16, 11, NSWidth([advParamBox frame])-32, NSHeight([advParamBox frame])-35)];
  }
}



- (void)addLogText:(NSString*)text
{
  [[logText textStorage] appendAttributedString:[[[NSAttributedString alloc] initWithString:text] autorelease]];
}


- (void)beginWork
{
  [sakila setImage:[NSImage imageNamed:@"dolphin_anim.gif"]];
  [sakila setAnimates:YES];
  
  _backWasEnabled= [backButton isEnabled];
  _nextWasEnabled= [nextButton isEnabled];

  [backButton setEnabled:NO];
  [nextButton setEnabled:NO];
  [cancelButton setEnabled:NO];

  [_grt setOutputHandler:self
                selector:@selector(addLogText:)];
  [_grt setMessageHandler:self
                 selector:@selector(addLogText:)];
}


- (void)endWork
{
  [_grt resetOutputHandler];
  [_grt resetMessageHandler];
  
  [sakila setAnimates:NO];
  [sakila setImage:[NSImage imageNamed:@"sakila.png"]];
 
  [backButton setEnabled:_backWasEnabled];
  [nextButton setEnabled:_nextWasEnabled];
  [cancelButton setEnabled:YES];
}


- (void)performSchemataFetches:(BOOL)goingBack
{
  const char *revEngModule;
  MYX_GRT_VALUE *result;
  
  MGRTValue connection([_connPanel writeConnectionToTarget]);

  [caption setStringValue:@"Source Schemata"];
  [description setStringValue:@"The list of available schemata is fetched."];

  revEngModule= connection["modules"]["ReverseEngineeringModule"].asString();
  
  [connImage setImage:_taskUnchecked];
  [schemaListImage setImage:_taskUnchecked];
  
  [fetchSchemaResult setHidden:YES];
  
  [tabView selectTabViewItemAtIndex:_section];

  [backButton setEnabled:YES];
  [nextButton setEnabled:YES];

  if (goingBack)
    return;
  
  [self beginWork];
  
  [_grt outText:@"Fetching schemata...\n"];

  if (_schemaList)
    delete _schemaList;
  _schemaList= 0;

  result= [_grt performModule:NSStr(revEngModule)
                     function:@"getSchemata"
                    arguments:[NSArray arrayWithObject:[NSValue valueWithPointer:[_grt globalValue:"/workbench/connection"]]]];
  if (result)
  {
    [connImage setImage:_taskChecked];
    [schemaListImage setImage:_taskChecked];
    [fetchSchemaResult setHidden:NO];
    
    [_grt outText:@"Schemata fetched.\n"];
    
    _schemaList= new MGRTValue(result);
  }
  else
  {
    [connImage setImage:_taskError];
    [schemaListImage setImage:_taskError];
    
    _nextWasEnabled= NO;

    [_grt outText:[NSString stringWithFormat:@"The list of schema names could not be retrieved (error: %i)\n%@\n",
      [_grt lastError], [_grt lastErrorDescription]]];
    if ([fetchSchemaAdvBox isHidden])
      [self showDetails:nil];
  }

  [self endWork];
}


- (void)refreshSchemataList
{
  [_schemaListObjects removeAllObjects];
  for (int i= 0; i < _schemaList->count(); i++)
  {
    [_schemaListObjects addObject:[NSString stringWithFormat:@"0%s", (*_schemaList)[i].asString()]];
  }
  
  
  [schemaSelectionList reloadData];
  
  [nextButton setEnabled:NO];
}


- (void)performSchemataSelection:(BOOL)goingBack
{
  MGRTValue connection([_connPanel writeConnectionToTarget]);

  [tabView selectTabViewItemAtIndex:_section];
  
  [caption setStringValue:@"Schema Selection"];
  [description setStringValue:@"Please select the schemata to reverse engineer."];
  
  [backButton setEnabled:YES];
  [nextButton setEnabled:YES];
  
  if (goingBack)
    return;
  
  [self refreshSchemataList];
  
  [[[[schemaSelectionList enclosingScrollView] superview] viewWithTag:1] setStringValue:[NSString stringWithFormat:@"%i items selected.", 0]];
}


- (BOOL)buildObjectSelection
{
  float y= 10;
  
  //Clear Filter Frames if there are some already
  for (int i= 0; i < [_filterPanels count]; i++)
    [[[_filterPanels objectAtIndex:i] topView] removeFromSuperview];
  [_filterPanels removeAllObjects];
  
  MGRTValue schemata((*_catalog)["schemata"]);
  
  //Get the struct type of the schemata
  MYX_GRT_STRUCT *gstruct= myx_grt_dict_struct_get([_grt grt], schemata[0].grtValue());
  
  //Clear the Mig.SourceObjects list
  MGRTValue sourceObjects(MGRTValue::fromGlobal([_grt grt], "/migration/sourceObjects"));
  sourceObjects.clear();
  
  //Create a filter frame for each member deriving from db.DatabaseObject
  for (int i= 0; i < myx_grt_struct_get_member_count_total([_grt grt], gstruct); i++)
  {
    MYX_GRT_STRUCT_MEMBER *member= myx_grt_struct_get_member_by_index_total([_grt grt], gstruct, i);
    if (member->value_type == MYX_STRING_VALUE || member->value_type == MYX_DICT_VALUE ||
        member->value_type == MYX_LIST_VALUE)
    {
      const char *memberName= myx_grt_struct_get_member_name(member);
      const char *contentStructName= myx_grt_struct_member_get_content_struct_name(member);
      
      //Only consider structs that derive from db.DatabaseObject
      if (contentStructName && 
          myx_grt_struct_inherits_from([_grt grt],contentStructName, "db.DatabaseObject"))
      {
        bool ok= false;
        //Collect all objects for the sourceObject list
        //Loop over all schemata
        for (int j= 0; j < schemata.count(); j++)
        {
          MGRTValue schema(schemata[j]);
          MGRTValue schemaItemList(schema[memberName]);
          
          if (schemaItemList.isValid())
          {
            //Add objects to sourceObject list
            for (int k= 0; k < schemaItemList.count(); k++)
            {
              MGRTValue schemaItem(schemaItemList[k]);
              sourceObjects.append(MGRTValue(schemaItem.dictId()));
              ok= true;
            }
          }
        }
                
        //Create FilterFrame (if there's content)
        if (ok)
        {
          NSView *view;
          MGRTRevEngFilterPane *filterFrame;
          
          filterFrame= [[MGRTRevEngFilterPane alloc] initWithMGRT:_grt forStruct:NSStr(contentStructName)];
          view= [filterFrame topView];
          
          [_filterPanels addObject:filterFrame];
          [view setAutoresizingMask:NSViewWidthSizable|NSViewMinYMargin];
          [view setFrameSize:NSMakeSize(NSWidth([_filterContent frame])-32, NSHeight([view frame]))];
          [view setFrameOrigin:NSMakePoint(16, y)];
          
          y+= NSHeight([view frame]) + 10;
          [_filterContent addSubview:view];
          
          if (g_str_has_suffix(contentStructName, ".Table")
              || g_str_has_suffix(contentStructName, ".View"))
          {
            [filterFrame setSelected:YES];
          }
          
          [view setPostsFrameChangedNotifications:YES];
        }
      }
    }
  }
  [_filterContent setFrameSize:NSMakeSize(NSWidth([_filterContent frame]), y)];
  
  for (int i= 0; i < [_filterPanels count]; i++)
    [[_filterPanels objectAtIndex:i] refreshLists];
  
  return YES;
}


- (void)performReverseEngineering:(BOOL)goingBack
{
  const char *revEngModule;
  MYX_GRT_VALUE *result;
  
  MGRTValue connection([_connPanel writeConnectionToTarget]);

  [caption setStringValue:@"Reverse Engineering"];
  [description setStringValue:@"The selected schemata is fetched and reverse engineered."];
  
  
  revEngModule= connection["modules"]["ReverseEngineeringModule"].asString();
  
  [revEngImage setImage:_taskUnchecked];
  [checkImage setImage:_taskUnchecked];
  [revEngResult setHidden:YES];
  
  [tabView selectTabViewItemAtIndex:_section];
  
  [backButton setEnabled:YES];
  
  if (goingBack)
  {
    [nextButton setEnabled:YES];
    return;
  }

  MGRTValue selectedSchemas(MGRTValue::createList(MYX_STRING_VALUE));
    
  [self beginWork];
  
  for (int i= 0; i < [_schemaListObjects count]; i++)
  {
    NSString *item= [_schemaListObjects objectAtIndex:i];
    if ([item hasPrefix:@"1"])
      selectedSchemas.append(MGRTValue([[item substringFromIndex:1] UTF8String]));
  }

  if (_catalog)
    delete _catalog;
  _catalog= 0;

  [_grt outText:@"Reverse engineering schema..."];
  
  result= [_grt performModule:NSStr(revEngModule)
                     function:@"reverseEngineer"
                    arguments:[NSArray arrayWithObjects:@"global::/workbench/connection",
                      [NSValue valueWithPointer:selectedSchemas.grtValue()], nil]];
  if (result)
  {
    _catalog = new MGRTValue(result);
    
    [_grt setGlobalValue:result
                 forPath:"/migration/sourceCatalog"];
  
    [revEngImage setImage:_taskChecked];
    
    if ([self buildObjectSelection])
    {
      [checkImage setImage:_taskChecked];
      
      [revEngResult setHidden:NO];
    }
    else
    {
      [checkImage setImage:_taskError];
      
    }
    [_grt outText:@"Reverse engineering finished.\n"];
  }
  else
  {
    [revEngImage setImage:_taskError];
    
    [_grt outText:[NSString stringWithFormat:@"The schema(ta) could not be reverse engineered (error: %i)\n%@\n",
      [_grt lastError], [_grt lastErrorDescription]]];
  }
  
  [self endWork];
}


- (void)performObjectSelection
{
  [caption setStringValue:@"Object Selection"];
  [description setStringValue:@"Add objects to the ignore list."];

  [tabView selectTabViewItemAtIndex:_section];
  
  [nextButton setEnabled:YES];
}


- (IBAction)changedPlacementOption:(id)sender
{
  if (sender == plAlgoPop)
  {
    [plOptions selectTabViewItemAtIndex:[sender indexOfSelectedItem]];
  }
}


- (void)performPlacementSelection
{
  [caption setStringValue:@"Placement Options"];
  [description setStringValue:@"Select options for placement of reverse engineered objects on the model."];
  
  [self changedPlacementOption:plAlgoPop];
  [tabView selectTabViewItemAtIndex:_section];
  
  [nextButton setEnabled:YES];
}


- (void)finalizeReverseEngineering
{
  [caption setStringValue:@"Finalization"];
  [description setStringValue:@"Finalizing the reverse engineering process."];

  [finRevEngImage setImage:_taskUnchecked];
  [finAddImage setImage:_taskUnchecked];
  if ([plAlgoPop indexOfSelectedItem] == 0)
    [finPlaceImage setImage:_taskDisabled];
  else
    [finPlaceImage setImage:_taskUnchecked];
  [finResult setStringValue:@""];
  [finResult setHidden:NO];

  [tabView selectTabViewItemAtIndex:_section];

  [nextButton setEnabled:NO];
  [backButton setEnabled:NO];
  [cancelButton setEnabled:YES];
  
  [[NSRunLoop currentRunLoop] runUntilDate:[NSDate dateWithTimeIntervalSinceNow:0.5]];

  [self beginWork];

  [_grt outText:@"Finalizing..."];
  [finResult setStringValue:@"Removing ignored objects from catalog"];

  [_grt performModule:@"DbUtils"
            procedure:@"removeIgnoredObjectsFromCatalog"
            arguments:[NSArray arrayWithObjects:[NSValue valueWithPointer:[_grt globalRefValue:"/migration/sourceCatalog"]], 
                                       [NSValue valueWithPointer:[_grt globalRefValue:"/migration/ignoreList"]], nil]];
  
  [_grt performModule:@"DbUtils"
            procedure:@"removeEmptySchemataFromCatalog"
            arguments:[NSArray arrayWithObject:[NSValue valueWithPointer:[_grt globalRefValue:"/workbench/catalog"]]]];

  [finRevEngImage setImage:_taskChecked];
    
  [_grt performAsyncModule:@"DbUtils"
                  function:@"mergeCatalogs"
                 arguments:[NSArray arrayWithObjects:
                   [NSValue valueWithPointer:[_grt globalRefValue:"/migration/sourceCatalog"]], 
                   [NSValue valueWithPointer:[_grt globalRefValue:"/workbench/catalog"]], nil]
          finishedSelector:@selector(mergeFinished:)
                    target:self
                  userData:nil];
}


- (void)arrangeSchemaFinished
{
  if (_arrangeSchemaError)
  {
    [finPlaceImage setImage:_taskError];
    [finResult setStringValue:@"Error during object placement."];
  }
  else
  {
    [finPlaceImage setImage:_taskChecked];
    [finResult setStringValue:@"Reverse engineering completed successfully."];
  }
  [_grt outText:@"Finished.\n"];
  
  [self endWork];
  
  if (_arrangeSchemaError)
  {
    [nextButton setEnabled:NO];
    [cancelButton setEnabled:YES];
  }
  else
  {
    [nextButton setEnabled:YES];
    [cancelButton setEnabled:NO];
  }
}


- (int)countObjectsToArrange
{
  MGRTValue schemaList(MGRTValue::fromGlobal([_grt grt], "/migration/sourceCatalog/schemata"));
  int count= 0;
  int c= schemaList.count();
  
  for (int i= 0; i < c; i++)
  {
    MGRTValue schema(schemaList[i]);
    MGRTValue list;

    for (int j= 0; j < schema.count(); j++)
    {
      const char *key;
      
      schema.dictItemByIndex(j, key, list);
      
      if (list.isValid() && list.type() == MYX_LIST_VALUE 
          && myx_grt_struct_inherits_from([_grt grt], list.listContentStruct(), "db.DatabaseObject"))
        count+= list.count();
    }
  }
  return count;
}


- (void)gridArrangeSchemas:(NSDictionary*)data
{
  MGRTValue model(MGRTValue::fromGlobal([_grt grt], "/workbench/model"));
  MGRTValue schemaList(MGRTValue::fromGlobal([_grt grt], "/migration/sourceCatalog/schemata"));
  int spacing, maxColumns;
  NSDictionary *userData= [data objectForKey:@"userData"];

  if (!userData)
  {
    int totalCount;
    _arrangeSchemaCount= 0;
    userData= data;
    
    totalCount= [self countObjectsToArrange];
    [progress setMaxValue:totalCount];
    [progress setMinValue:0];
    [progress setDoubleValue:0.0];
    //[progress setHidden:NO];    
  }
  else
  {
    if (data && [data objectForKey:@"error"])
    {
      _arrangeSchemaError= YES;
      [_grt outText:[data objectForKey:@"message"]];
    }    
    _arrangeSchemaCount++;
    [progress setDoubleValue:_arrangeSchemaCount];
  }
  spacing= [[userData objectForKey:@"spacing"] intValue];
  maxColumns= [[userData objectForKey:@"maxColumns"] intValue];

  
  if (_arrangeSchemaCount >= schemaList.count())
  {
    [self arrangeSchemaFinished];
  }
  else
  {
    MGRTValue schema(schemaList[_arrangeSchemaCount]);
    
    //check if the current view is empty
    MGRTValue currentView(MGRTValue::refObject([_grt grt], model["currentView"].asString()));
    
    MGRTValue viewElementList(currentView["elements"]);
    if (viewElementList.count() > 0)
    {
      // if not, add a new view
      [_grt performAsyncModule:@"Workbench"
                      function:@"arrangeSchemaOnNewView"
                     arguments:[NSArray arrayWithObjects:
                       [NSValue valueWithPointer:model.grtValue()],
                       NSStr(schema["name"].asString()),     
                       [NSGRTValue grtValueWithValue:schema.grtValue()],
                       [NSNumber numberWithInt:20],
                       [NSNumber numberWithInt:20],
                       [NSNumber numberWithInt:maxColumns],
                       [NSNumber numberWithInt:spacing],
                       nil]
              finishedSelector:@selector(gridArrangeSchemas:)
                        target:self
                      userData:userData];
    }
    else
    {
      [_grt performAsyncModule:@"Workbench"
                      function:@"arrangeSchemaOnCurrentView"
                     arguments:[NSArray arrayWithObjects:
                       [NSGRTValue grtValueWithValue:schema.grtValue()],
                       [NSNumber numberWithInt:20],
                       [NSNumber numberWithInt:20],
                       [NSNumber numberWithInt:maxColumns],
                       [NSNumber numberWithInt:spacing],
                       nil]
              finishedSelector:@selector(gridArrangeSchemas:)
                        target:self
                      userData:userData];
    }
  }
}


- (void)autoArrangeSchemas:(NSDictionary*)data
{  
  MGRTValue model(MGRTValue::fromGlobal([_grt grt], "/workbench/model"));
  MGRTValue schemaList(MGRTValue::fromGlobal([_grt grt], "/migration/sourceCatalog/schemata"));

  if (data && [data objectForKey:@"error"])
  {
    _arrangeSchemaError= YES;
    [_grt outText:[data objectForKey:@"message"]];
  }
  
  if (!data)
  {
    int totalCount;
    _arrangeSchemaCount= 0;
    totalCount= [self countObjectsToArrange];
    [progress setMaxValue:totalCount];
    [progress setMinValue:0];
    [progress setDoubleValue:0.0];
    //[progress setHidden:NO];
  }
  else
  {
    _arrangeSchemaCount++;
    [progress setDoubleValue:_arrangeSchemaCount];
  }

  if (_arrangeSchemaCount >= schemaList.count())
  {
    [progress setHidden:YES];
    [self arrangeSchemaFinished];
  }
  else
  {
    MGRTValue schema(schemaList[_arrangeSchemaCount]);

    //check if the current view is empty
    MGRTValue currentView(MGRTValue::refObject([_grt grt], model["currentView"].asString()));

    MGRTValue viewElementList(currentView["elements"]);
    if (viewElementList.count() > 0)
    {
      // if not, add a new view
      [_grt performAsyncModule:@"Workbench"
                      function:@"loadAndAutoArrangeInNewView"
                     arguments:[NSArray arrayWithObjects:
                       [NSGRTValue grtValueWithValue:schema.grtValue()],
                       NSStr(schema["name"].asString()),
                       nil]
              finishedSelector:@selector(autoArrangeSchemas:)
                        target:self
                      userData:nil];
    }
    else
    {
      [_grt performAsyncModule:@"Workbench"
                      function:@"loadAndAutoArrange"
                     arguments:[NSArray arrayWithObjects:
                       [NSGRTValue grtValueWithValue:schema.grtValue()],
                       [NSGRTValue grtValueWithValue:model["currentView"].grtValue()],
                       nil]
              finishedSelector:@selector(autoArrangeSchemas:)
                        target:self
                      userData:nil];
    }
  }
}


- (void)mergeFinished:(NSDictionary*)data
{  
  if ([data objectForKey:@"error"])
  {
    [_grt outText:@"Operation failed during catalog merge."];
    [_grt outText:[data objectForKey:@"message"]];
    [finRevEngImage setImage:_taskError];
    [self endWork];
    
    [nextButton setEnabled:NO];
    [backButton setEnabled:NO];
    [cancelButton setEnabled:YES];
    return;
  }
  
  [finAddImage setImage:_taskChecked];
  
  switch ([plAlgoPop indexOfSelectedItem])
  {
    case 1:
    {
      int maxColumns;
      int spacing;
      
      switch ([plSpacePop indexOfSelectedItem])
      {
        default:
        case 0: spacing= 40; break;
        case 1: spacing= 70; break;
        case 2: spacing= 100; break;
      }
      
      switch ([plWidthPop indexOfSelectedItem])
      {
        default:
        case 0: maxColumns= 5; break;
        case 1: maxColumns= 10; break;
        case 2: maxColumns= 15; break;
        case 3: maxColumns= 20; break;
        case 4: maxColumns= 30; break;
      }
      
      [finResult setStringValue:@"Placing objects on the canvas"];
      [self gridArrangeSchemas:[NSDictionary dictionaryWithObjectsAndKeys:
        [NSNumber numberWithInt:spacing], @"spacing",
        [NSNumber numberWithInt:maxColumns], @"maxColumns",
        nil]];
      break;
    }
    case 2:
    {
      [finResult setStringValue:@"Placing objects on the canvas"];
      [self autoArrangeSchemas:nil];
      break;
    }
  }  
}


- (void)updateSection:(BOOL)goingBack
{
  if (_section == 0)
    [backButton setEnabled:NO];
  else 
    [backButton setEnabled:YES];

  if (_section == 6)
    [nextButton setTitle:@"Finish"];
  else
    [nextButton setTitle:@"Next >"];

  
  NSView *advView= nil;  
  switch (_section)
  {
    case 0: advView= advParamBox; break;
    case 1:      
      advView= fetchSchemaAdvBox;
      [[logText enclosingScrollView] retain];
      [[logText enclosingScrollView] removeFromSuperview];
      [advView addSubview:[logText enclosingScrollView]];
      [[logText enclosingScrollView] release];
      break;
    case 3:
      advView= revEngAdvBox;
      [[logText enclosingScrollView] retain];
      [[logText enclosingScrollView] removeFromSuperview];
      [advView addSubview:[logText enclosingScrollView]];
      [[logText enclosingScrollView] release];
      break;
    case 6:
      advView= finAdvBox; 
      [[logText enclosingScrollView] retain];
      [[logText enclosingScrollView] removeFromSuperview];
      [advView addSubview:[logText enclosingScrollView]];
      [[logText enclosingScrollView] release];
      break;
  }
  if (advView)
    [detailsButton setEnabled:YES];
  else
    [detailsButton setEnabled:NO];
  
  switch (_section)
  {
    case 0:
      [caption setStringValue:@"Source Connection"];
      [description setStringValue:@"Please specify the source connection."];
      [tabView selectTabViewItemAtIndex:_section];
      [nextButton setEnabled:YES];
      break;
    case 1:
      [self performSchemataFetches:goingBack];
      break;
    case 2:
      [self performSchemataSelection:goingBack];
      break;
    case 3:
      [self performReverseEngineering:goingBack];
      break;
    case 4:
      [self performObjectSelection];
      break;
    case 5:
      [self performPlacementSelection];
      break;
    case 6:
      [self finalizeReverseEngineering];
      break;      
  }
}


- (IBAction)goBack:(id)sender
{
  if (_section > 0)
  {
    _section--;
    [self updateSection:YES];
  }
}


- (IBAction)goNext:(id)sender
{
  if (_section < 6)
  {
    _section++;
    [self updateSection:NO];
  }
  else if (_section == 6)
  {
    [NSApp abortModal];
  }
}


- (IBAction)cancel:(id)sender
{
  [NSApp abortModal];
}


- (IBAction)showDetails:(id)sender
{
  _advancedShown= !_advancedShown;
  if (_advancedShown)
    [detailsButton setTitle:@"Show Less"];
  else
    [detailsButton setTitle:@"Show More"];

  [advParamBox setHidden:!_advancedShown];
  [fetchSchemaAdvBox setHidden:!_advancedShown];
  [revEngAdvBox setHidden:!_advancedShown];
  [finAdvBox setHidden:!_advancedShown];  
}



- (void)grtMessageHandler:(NSString*)message type:(NSString*)type
{
  NSLog(@"%@ // %@", message, type);
}



- (BOOL)run
{
  [_grt setMessageHandler:self selector:@selector(grtMessageHandler:type:)];
  [[self window] makeKeyAndOrderFront:nil];
  [self updateSection:NO];
  [NSApp runModalForWindow:[self window]];
  [self close];
  [_grt resetMessageHandler];
  return NO;
}


- (int)numberOfRowsInTableView:(NSTableView *)aTableView
{
  if (aTableView == schemaSelectionList)
  {
    return [_schemaListObjects count];
  }
  return 0;
}


- (void)tableView:(NSTableView *)aTableView setObjectValue:(id)anObject forTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
  if (aTableView == schemaSelectionList)
  {
    if ([anObject intValue] == NSOnState)
      [_schemaListObjects replaceObjectAtIndex:rowIndex
                                    withObject:[@"1" stringByAppendingString:[[_schemaListObjects objectAtIndex:rowIndex] substringFromIndex:1]]];
    else
      [_schemaListObjects replaceObjectAtIndex:rowIndex
                                    withObject:[@"0" stringByAppendingString:[[_schemaListObjects objectAtIndex:rowIndex] substringFromIndex:1]]];        
  }
  
  int count= 0;
  for (int i= 0; i < [_schemaListObjects count]; i++)
  {
    if ([[_schemaListObjects objectAtIndex:i] hasPrefix:@"1"])
      count++;
  }
  [[[[schemaSelectionList enclosingScrollView] superview] viewWithTag:1] setStringValue:[NSString stringWithFormat:@"%i items selected.", count]];
  
  [nextButton setEnabled:count > 0];
}


- (id)tableView:(NSTableView *)aTableView objectValueForTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
  if (aTableView == schemaSelectionList)
  {
    return [NSNumber numberWithInt:[[_schemaListObjects objectAtIndex:rowIndex] hasPrefix:@"1"] ? NSOnState : NSOffState];
  }
  return nil;
}


- (void)tableView:(NSTableView *)aTableView willDisplayCell:(id)aCell forTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
  if (aTableView == schemaSelectionList)
  {
    [aCell setTitle:[[_schemaListObjects objectAtIndex:rowIndex] substringFromIndex:1]];
  }
}


@end
