unit PrintProgress;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ComCtrls, ExtCtrls,
  GenericCanvasViewer, ModelPrintSetup;

type
  TModelPrintForm = class(TForm)
    InfoLabel: TLabel;
    ProgressBar: TProgressBar;
    CancelButton: TButton;
    Label1: TLabel;
    Label2: TLabel;
    PageLabel: TLabel;
    Image1: TImage;
  private
  public
  end;

procedure CanvasPrint(Viewer: TGenericCanvasViewer; HPageCount, VPageCount: Integer; Selection: TPageSelection);

//----------------------------------------------------------------------------------------------------------------------

implementation

{$R *.dfm}

uses
  Printers;

//----------------------------------------------------------------------------------------------------------------------

procedure CanvasPrint(Viewer: TGenericCanvasViewer; HPageCount, VPageCount: Integer; Selection: TPageSelection);

// Prints the canvas to the currently active printer. The canvas is tiled over HPageCount times VPageCount pages.

var
  TargetSizeX: Integer;
  TargetSizeY: Integer;
  SourceSizeX: Integer;
  SourceSizeY: Integer;
  PrintZoom: Single;
  Bounds: TViewport;
  Aborted: Boolean;
  Buffer: TBitmap;
  Memory: Pointer;

  Current: Integer;
  Printed: Integer;
  Total: Integer;
  Progress: TModelPrintForm;

  //----------------------------------------------------------------------------

  procedure AdjustBuffer(NewHeight, NewWidth: Integer);

  // Resizes the buffer bitmap to the given size if it differs and
  // retrieves the pointer to the first pixel, which is then written to Memory.

  begin
    if (Buffer.Height <> NewHeight) or (Buffer.Width <> NewWidth) then
    begin
      // Set image height to 0 to avoid copy of a large memory block when reallocating the image size.
      Buffer.Height := 0;

      Buffer.Width := NewWidth;
      Buffer.Height := NewHeight;

      if Buffer.Height > 0 then
      begin
        Memory := Buffer.ScanLine[0];
        if Cardinal(Buffer.ScanLine[1]) < Cardinal(Memory) then
          Memory := Buffer.ScanLine[Buffer.Height - 1];
      end;
    end;
  end;

  //----------------------------------------------------------------------------

  function DoProgress: Boolean;

  // Update the progress on the progress form and checks if the user has
  // cancelled the operation.

  begin
    Application.ProcessMessages;
    Result := Progress.ModalResult <> mrCancel;
    if Result then
    begin
      Progress.PageLabel.Caption := Format('page %d is being printed (%d of %d)',
        [Current, Printed, Total]);
      Progress.ProgressBar.Position := Printed;
      Progress.Update;
    end;
  end;

  //----------------------------------------------------------------------------

var
  I: Integer;

begin
  Application.ProcessMessages;
  Screen.Cursor := crHourGlass;
  Progress := TModelPrintForm.Create(nil);
  try
    Current := 0;
    Printed := 1;
    Total := 0;

    // Count total pages to print. Pages *not* to print are marked with True.
    for I := 0 to Length(Selection) - 1 do
      if not Selection[I] then
        Inc(Total);
    Progress.FormStyle := fsStayOnTop;
    Progress.ProgressBar.Max := Total;
    Aborted := not DoProgress;
    Progress.Show;
    Application.ProcessMessages;

    TargetSizeX := HPageCount * Printer.PageWidth;
    TargetSizeY := VPageCount * Printer.PageHeight;

    // Selection of the print area tiling is done so that at least one of both directions fits entirely.
    // That means that either horizontally or vertically the canvas fits exactly on the target print area.
    // The larger direction is fitted here.
    if TargetSizeX > TargetSizeY then
    begin
      PrintZoom := TargetSizeX / Viewer.BaseSizeX;
      SourceSizeX := Round(PrintZoom * Viewer.BaseSizeX);

      // The aspect ratio of the target canvas must of course also be used for the source canvas.
      SourceSizeY := Round(SourceSizeX * TargetSizeY / TargetSizeX);
    end
    else
    begin
      PrintZoom := TargetSizeY / Viewer.BaseSizeY;
      SourceSizeY := Round(PrintZoom * Viewer.BaseSizeY);
      SourceSizeX := Round(SourceSizeY * TargetSizeX / TargetSizeY);
    end;

    Bounds.Top := 0;
    Bounds.Width := Printer.PageWidth;
    Bounds.Height := Printer.PageHeight;

    Buffer := TBitmap.Create;
    try
      Buffer.PixelFormat := pf32Bit;
      while not Aborted and (Bounds.Top < SourceSizeY) do
      begin
        Bounds.Left := 0;

        // In the last run we likely don't need the full buffer height. Adjust the bounds height accordingly.
        if (SourceSizeY - Bounds.Top) < Buffer.Height then
        begin
          AdjustBuffer(SourceSizeY - Bounds.Top, Printer.PageWidth);
          Bounds.Height := Buffer.Height;
        end
        else
          AdjustBuffer(Printer.PageHeight, Printer.PageWidth);

        while not Aborted and (Bounds.Left < SourceSizeX) do
        begin
          Inc(Current);

          // In the last run we likely don't need the full buffer height. Adjust the bounds height accordingly.
          if (SourceSizeX - Bounds.Left) < Buffer.Width then
          begin
            AdjustBuffer(Buffer.Height, SourceSizeX - Bounds.Left);
            Bounds.Width := Buffer.Width;
          end;

          // Check if the current page should be printed at all.
          // -1 because we already increased the value of Current above.
          // not Selection because a value of True means to omit that page.
          if not Selection[Current - 1] then
          begin
            Viewer.Canvas.RenderToMemory(Memory, GC_COLOR_FORMAT_BGRA, GC_RENDER_GRID or GC_RENDER_CONNECTIONS or
              GC_RENDER_FEEDBACK, PrintZoom, Bounds);

            Printer.Title := 'MySQL Workbench - Model print page ' + IntToStr(Current);
            Printer.BeginDoc;
            Printer.Canvas.Draw(0, 0, Buffer);
            Printer.EndDoc;
            Inc(Printed);
          end;

          Bounds.Left := Bounds.Left + Printer.PageWidth;

          Aborted := not DoProgress;
        end;

        Bounds.Top := Bounds.Top + Printer.PageHeight;
      end;
    finally
      Buffer.Free;
    end;
  finally
    Progress.Free;
    Screen.Cursor := crDefault;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

end.

