/*=========================================================================
| Open Office QuickStarter
|--------------------------------------------------------------------------
| (c) 2002  Kumaran Santhanam  <kumaran@alumni.stanford.org>
|
| This project is released under the GNU General Public License.
| Please see the file COPYING for more details.
|--------------------------------------------------------------------------
| gnome.cxx
|
| This file contains the GNOME panel applet front-end.
 ========================================================================*/

/*=========================================================================
| INCLUDES
 ========================================================================*/
extern "C" {
#include <applet-widget.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <X11/SM/SMlib.h>
}

#include "model.h"
#include "ooqstart-on.xpm"
#include "ooqstart-wait.xpm"
#include "ooqstart-off.xpm"


/*=========================================================================
| CONSTANTS
 ========================================================================*/
#define PACKAGE  "gnome-applets"


/*=========================================================================
| STATIC GLOBALS
 ========================================================================*/
static GtkWidget *propwindow;
static GtkWidget *applet;
static GtkWidget *entry;
static GtkWidget *checkbox;

static GtkWidget *tile;
static GtkWidget *tilepixmap;

static guint      timeout = 0;

static GnomeClient *client;
static char        *sessionHandle = 0;

enum IconState { ICON_NONE, ICON_OFF, ICON_WAIT, ICON_ON };
static IconState   iconState = ICON_NONE;


/*=========================================================================
| LAUNCHERS
 ========================================================================*/
static void startWriter (AppletWidget *, gpointer pModel) {
    ((Model *)pModel)->startWriter();
}
static void startCalc (AppletWidget *, gpointer pModel) {
    ((Model *)pModel)->startCalc();
}
static void startDraw (AppletWidget *, gpointer pModel) {
    ((Model *)pModel)->startDraw();
}
static void startImpress (AppletWidget *, gpointer pModel) {
    ((Model *)pModel)->startImpress();
}


/*=========================================================================
| UTILITY FUNCTIONS
 ========================================================================*/
// Create a pixmap from character data
static GtkWidget *create_pixmap (GtkWidget *widget, char **pixmap_char,
                                 int width, int height)
{
    GdkPixbuf *pixbuf, *tilepixbuf;

    pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)pixmap_char);
    if (pixbuf == 0) {
        // This should never happen, since the pixmap
        // is compiled into the code.
        g_warning("Error loading pixmap.");
        return 0;
    }

    tilepixbuf = gdk_pixbuf_scale_simple(pixbuf, width, height,
                                         GDK_INTERP_BILINEAR);

    GdkPixmap   *gdkpixmap;
    GdkBitmap   *mask;
    GtkWidget   *pixmap;
    gdk_pixbuf_render_pixmap_and_mask(tilepixbuf, &gdkpixmap, &mask, 127);
    
    pixmap = gtk_pixmap_new(gdkpixmap, mask);

    gdk_bitmap_unref(mask);
    gdk_pixmap_unref(gdkpixmap);
    gdk_pixbuf_unref(pixbuf);
    gdk_pixbuf_unref(tilepixbuf);

    return pixmap;
}


/*=========================================================================
| GUI UPDATE
 ========================================================================*/
static void updateGUI (Model *model) {
    IconState state = ICON_OFF;

    switch (model->getStatus()) {
      case MODEL_OFFICE_STOPPED:   state = ICON_OFF;  break;
      case MODEL_OFFICE_STARTING:  state = ICON_WAIT; break;
      case MODEL_OFFICE_RUNNING:   state = ICON_ON;   break;
    }

    if (state != iconState) {
        int panelPixelSize =
            applet_widget_get_panel_pixel_size(APPLET_WIDGET(applet));
        gtk_widget_set_usize(tile, panelPixelSize, panelPixelSize);

        bool en = (state == ICON_ON);
        applet_widget_callback_set_sensitive(APPLET_WIDGET(applet), "sw", en);
        applet_widget_callback_set_sensitive(APPLET_WIDGET(applet), "sc", en);
        applet_widget_callback_set_sensitive(APPLET_WIDGET(applet), "si", en);
        applet_widget_callback_set_sensitive(APPLET_WIDGET(applet), "sd", en);

        if (tilepixmap) gtk_container_remove(GTK_CONTAINER(tile), tilepixmap);
        tilepixmap = 0;

        char **pixmapData = 0;
        switch (state) {
          case ICON_OFF:   pixmapData = ooqstart_off_xpm;   break;
          case ICON_WAIT:  pixmapData = ooqstart_wait_xpm;  break;
          case ICON_ON:    pixmapData = ooqstart_on_xpm;    break;
          default:         pixmapData = 0;                  break;
        }
        
        if (pixmapData) {
            tilepixmap = create_pixmap(GTK_WIDGET(tile), pixmapData,
                                       panelPixelSize, panelPixelSize);
        }
        
        if (tilepixmap) {
            gtk_widget_show(tilepixmap);
            gtk_container_add(GTK_CONTAINER(tile), tilepixmap);
        }
        
        iconState = state;
    }
}


/*=========================================================================
| SESSION MANAGEMENT
 ========================================================================*/
// These functions are necessary to disable GNOME session management
// for the Open Office binary.  Specifically, the binary should not
// be relaunched upon login.
// 
// Session management is rather nasty.  Please see the GNOME documentation
// for more information.  A lot of this code looks unnecessary, but
// unfortunately, it is required because of the quirks in the X11
// session management protocol.
#define GsmClientEvent           "_GSM_ClientEvent"
#define GsmCommand               "_GSM_Command"
#define GsmSelectClientEvents    "SelectClientEvents"
#define GsmDeselectClientEvents  "DeselectClientEvents"
#define GsmChangeProperties      "ChangeProperties"

static SmProp *mkSmProp (gchar *name, gchar *value1, gchar *value2) {
    SmProp *prop = (SmProp *)malloc(sizeof(SmProp));

    prop->name     = strdup(name);
    prop->type     = strdup(SmLISTofARRAY8);
    prop->num_vals = 0;
    if (value1)  ++prop->num_vals;
    if (value2)  ++prop->num_vals;
    
    prop->vals = (SmPropValue *)malloc(prop->num_vals * sizeof(SmPropValue));
    for (int i = 0; i < prop->num_vals; ++i) {
        gchar *value = (i == 0) ? value1 : value2;
        prop->vals[i].value  = value;
        prop->vals[i].length = strlen(value);
    }
    
    return prop;
}

static SmProp *mkSmProp (gchar *name, gchar *value) {
    return mkSmProp(name, value, 0);
}

static SmProp *mkSmProp (gchar *name, int value) {
    SmProp *prop = (SmProp *)malloc(sizeof(SmProp));

    prop->name     = strdup(name);
    prop->type     = strdup(SmCARD8);
    prop->num_vals = 1;
    prop->vals     = (SmPropValue *)malloc(sizeof(SmPropValue));
    prop->vals[0].value  = strdup("X");
    prop->vals[0].length = 1;
    ((char *)prop->vals[0].value)[0] = (gchar)value;
    
    return prop;
}

static void disableRestart (gchar *handle)
{
    SmProp *props[2];
    props[0] = mkSmProp(GsmCommand, GsmChangeProperties, handle);
    props[1] = mkSmProp(SmRestartStyleHint, SmRestartNever);
    SmcSetProperties((SmcConn)client->smc_conn, 2, props);
}

static gint makeRequest (gpointer data);

static void smPropsCallback (SmcConn smc_conn, SmPointer data,
                             int nprops, SmProp **props)
{
    const char *program = ((Model *)data)->getOpenOfficeProgram();
    if (program == 0)  goto free_props;
    if (nprops  <= 0)  goto free_props;

    // Check if this is a client event
    if (strcmp(props[0]->name, GsmClientEvent) == 0) {
        char *handle = (char *)props[0]->vals[1].value;
        // Go through each property to find a Program path
        for (int i = 1; i < nprops; ++i) {
            if (props[i]->name                    != 0 &&
                strcmp(props[i]->name, "Program") == 0 &&
                props[i]->num_vals                == 1)
            {
                // Check to see if the name matches
                char *val = (char *)props[i]->vals->value;
                if (strcmp(program, val) == 0) {
                    // If we got this far, we have found the
                    // correct client.
                    printf("Open Office session %s = %s\n", handle, val);
                    if (sessionHandle)  free(sessionHandle);
                    sessionHandle = strdup(handle);
                    disableRestart(sessionHandle);
                }
            }
        }
    }

  free_props:
    for (int i = 0; i < nprops; ++i) {
        if (props[i])  SmFreeProperty(props[i]);
    }
    free(props);

    // This is a good time to update the GUI, since it will
    // give a snappy response to the user.
    updateGUI((Model *)data);
    
    // Queue the next request.  We can't do it here because nested
    // calls to SmcGetProperties messes up the session manager.
    // Instead, we'll call it from the GTK idle loop.
    gtk_idle_add(makeRequest, data);
}


static void disableSessionManagement (Model *model) {
    // This is tricky because the X11 SM works on a callback
    // basis.  Essentially, what this function does is
    // request all of the session manager's properties.
    //
    // The request function takes a callback which is
    // called as the properties come in one by one.
    // 
    // If the callback finds a match between a particular
    // entry and the given program name, it will disable
    // session management for the program.
    SmProp *props[1];
    props[0] = mkSmProp(GsmCommand, GsmSelectClientEvents);
    SmcSetProperties((SmcConn)client->smc_conn, 1, &props[0]);
    gtk_idle_add(makeRequest, model);
}

static gint makeRequest (gpointer data) {
    SmcGetProperties((SmcConn)client->smc_conn, smPropsCallback, data);
    return FALSE;
}


/*=========================================================================
| LOAD / SAVE CONFIGURATION
 ========================================================================*/
static void loadConfig (char *cfgpath, Model *model) {
    gnome_config_push_prefix(cfgpath);
       
    model->setOpenOfficeDir(gnome_config_get_string("ooqstart/openoffice"));
    if (gnome_config_get_bool("ooqstart/enable=false"))
        model->enable();
    else
        model->disable();
    
    gnome_config_pop_prefix();
}

static void saveConfig (char *cfgpath, Model *model) {
    gnome_config_push_prefix(cfgpath);
    gnome_config_set_string("ooqstart/openoffice", model->getOpenOfficeDir());
    gnome_config_set_bool("ooqstart/enable", model->getEnabled());
    gnome_config_pop_prefix();

    gnome_config_sync();
}


/*=========================================================================
| PROPERTIES / ABOUT
 ========================================================================*/
static void propApply (GtkWidget *widget, gint page, gpointer data)
{
    if (propwindow == 0)  return;
    
    Model *model = (Model *)data;
    
    // Get the status of the checkbox
    bool enable = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbox));

    // Check to see if the given path is valid
    model->setOpenOfficeDir(gtk_entry_get_text(GTK_ENTRY(entry)));
    if (!model->isValid()) {
        GtkWidget *dialog = gnome_message_box_new
            (_("Open Office was not found in the given path.\n"
               "The QuickStarter applet will be disabled.\n"),
             GNOME_MESSAGE_BOX_WARNING,
             GNOME_STOCK_BUTTON_OK,
             0);
        gnome_dialog_run(GNOME_DIALOG(dialog));
        
        // Disable the quick starter
        enable = false;
    }

    if (enable)
        model->enable();
    else
        model->disable();

    saveConfig(APPLET_WIDGET(applet)->privcfgpath, model);
    applet_widget_sync_config(APPLET_WIDGET(applet));
    propwindow = 0;
}

static void showHelp (AppletWidget *, gpointer) {
    GnomeHelpMenuEntry entry = { "ooqstart_applet", "index.html" };
    gnome_help_display(0, &entry);
}

static gint propDestroy (GtkWidget *, gpointer) {
    return 0;
}

static void properties (AppletWidget *, gpointer pModel) {
    Model *model = (Model *)pModel;

    propwindow = gnome_property_box_new();
    gtk_window_set_title
        (GTK_WINDOW(&GNOME_PROPERTY_BOX(propwindow)->dialog.window),
         _("Open Office QuickStarter Applet Settings"));

    GtkWidget *vbox = gtk_vbox_new(FALSE, GNOME_PAD_SMALL);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), GNOME_PAD_SMALL);
    
    GtkWidget *label = gtk_label_new
        (_("Full path to Open Office or Star Office base directory:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
    gtk_widget_show(label);

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), model->getOpenOfficeDir());
    gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);
    GTK_WIDGET_SET_FLAGS(entry, GTK_CAN_FOCUS);
    gtk_signal_connect_object
        (GTK_OBJECT(entry), "changed",
         GTK_SIGNAL_FUNC(gnome_property_box_changed),
         GTK_OBJECT(propwindow));
    gtk_widget_show(entry);
    
    label = gtk_label_new(_(""));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
    gtk_widget_show(label);
    
    checkbox = gtk_check_button_new_with_label
        (_("Enable QuickStarter"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox),
                                 model->getEnabled());
    gtk_box_pack_start(GTK_BOX(vbox), checkbox, FALSE, FALSE, 0);
    gtk_signal_connect_object
        (GTK_OBJECT(checkbox), "clicked",
         GTK_SIGNAL_FUNC(gnome_property_box_changed),
         GTK_OBJECT(propwindow));
    gtk_widget_show(checkbox);

    label = gtk_label_new(_("General"));
    gnome_property_box_append_page
        (GNOME_PROPERTY_BOX(propwindow), vbox, label);
    gtk_widget_show(vbox);

    gtk_signal_connect(GTK_OBJECT(propwindow), "apply",
                       GTK_SIGNAL_FUNC(propApply), pModel);
    gtk_signal_connect(GTK_OBJECT(propwindow), "destroy",
                       GTK_SIGNAL_FUNC(propDestroy), 0);
    gtk_signal_connect(GTK_OBJECT(propwindow), "help",
                       GTK_SIGNAL_FUNC(showHelp),    0);
    gtk_widget_show_all(propwindow);
}
    
static void about (AppletWidget *, gpointer) {
    static GtkWidget *about = 0;

    if (about == 0) {
        const gchar *authors[4];
        authors[0] = _("Kumaran Santhanam <kumaran@alumni.stanford.org>");
        authors[1] = 0;

        about = gnome_about_new
            ( _("Open Office QuickStarter Applet"), VERSION,
              _("(C) 2002 Kumaran Santhanam"), authors,
              _("Quick starter launch applet for Open Office 641C+ "
                "and Star Office 6.0+.\n"
                "Right-click on the applet icon for a context menu.\n"
                " \n"
                "Released under the GNU General Public License v2."),
              0);

        gtk_signal_connect(GTK_OBJECT(about), "destroy",
                           GTK_SIGNAL_FUNC(gtk_widget_destroyed), &about);
        gtk_widget_show(about);
    }

    gdk_window_show(about->window);
    gdk_window_raise(about->window);
}


/*=========================================================================
| MENUS
 ========================================================================*/
static void menuItem (GtkWidget *applet,
                      char *name, char *text, char *stock,
                      void (*func) (AppletWidget *, gpointer),
                      Model *model)
{
    if (stock == 0)
        applet_widget_register_callback
            (APPLET_WIDGET(applet), name, _(text), func, model);
    else
        applet_widget_register_stock_callback
            (APPLET_WIDGET(applet), name, stock, _(text), func, model);
}

static void tileClicked (GtkWidget *w, gpointer data) {
    about((AppletWidget *)data, 0);
}


/*=========================================================================
| BACKGROUND TIMER
 ========================================================================*/
static gint timeoutFunc (gpointer pModel) {
    Model *model = (Model *)pModel;
    model->process();
    updateGUI(model);
    return TRUE;
}


/*=========================================================================
| APPLET CALLBACKS
 ========================================================================*/
static void resizeApplet (GtkWidget *applet, gint arg1, gpointer data) {
    iconState = ICON_NONE;
    updateGUI((Model *)data);
}

static gint saveSession (GtkWidget *applet, gchar *privcfgpath,
                         gchar *globcfgpath, gpointer data)
{
    saveConfig(privcfgpath, (Model *)data);
    gnome_config_drop_all();
    return FALSE;
}

static void destroyApplet (GtkWidget *applet, gpointer data) {
    if (timeout)  gtk_timeout_remove(timeout);

    // Deselect client events and disconnect from the
    // session manager.  If this isn't done, the session
    // manager crashes, taking all of GNOME with it.
    SmProp *props[1];
    props[0] = mkSmProp(GsmCommand, GsmDeselectClientEvents);
    SmcSetProperties((SmcConn)client->smc_conn, 1, props);
    
    gnome_client_disconnect(client);
}


/*=========================================================================
| MAIN PROGRAM
 ========================================================================*/
int main (int argc, char **argv)
{
    // Initialize the i18n stuff
    (void)bindtextdomain(PACKAGE, GNOMELOCALEDIR);
    (void)textdomain(PACKAGE);

    // intialize, this will basically set up the applet, corba and
    // call gnome_init
    applet_widget_init("ooqstart_applet", VERSION, argc, argv, NULL, 0, NULL);
    
    // create a new applet_widget
    applet = applet_widget_new("ooqstart_applet");

    // in the rare case that the communication with the panel
    // failed, error out
    if (!applet)  g_error("Can't create applet!\n");

    // Get a GNOME client connected to the session manager
    client = gnome_client_new();
    gnome_client_connect(client);

    // create the widget we are going to put on the applet
    tile = gtk_button_new();
    gtk_button_set_relief(GTK_BUTTON(tile), GTK_RELIEF_NONE);
    GTK_WIDGET_UNSET_FLAGS(tile, GTK_CAN_FOCUS);
    gtk_widget_show(tile);
    gtk_signal_connect(GTK_OBJECT(tile), "clicked",
                       GTK_SIGNAL_FUNC(tileClicked), applet);

    // add the widget to the applet-widget, and thereby actually
    // putting it "onto" the panel
    applet_widget_add(APPLET_WIDGET(applet), tile);
    gtk_widget_show(applet);

    // initialize the model
    Model model;

    // connect the applet callbacks
    gtk_signal_connect(GTK_OBJECT(applet), "change_pixel_size",
                       GTK_SIGNAL_FUNC(resizeApplet), &model);
    gtk_signal_connect(GTK_OBJECT(applet), "save_session",
                       GTK_SIGNAL_FUNC(saveSession), &model);
    gtk_signal_connect(GTK_OBJECT(applet), "destroy",
                       GTK_SIGNAL_FUNC(destroyApplet), &model);

    // add the items to the applet menu
    menuItem(applet, "props",   "Properties...",
             GNOME_STOCK_PIXMAP_PROPERTIES,
             properties, &model);
    menuItem(applet, "about",   "About...",
             GNOME_STOCK_PIXMAP_ABOUT,
             about, 0);
    menuItem(applet, "help",    "Help",
             GNOME_STOCK_PIXMAP_HELP,
             showHelp, 0);

    char *icon = 0;
    menuItem(applet, "sw",  "New Text Document", icon, startWriter,  &model);
    menuItem(applet, "sc",  "New Spreadsheet",   icon, startCalc,    &model);
    menuItem(applet, "si",  "New Presentation",  icon, startImpress, &model);
    menuItem(applet, "sd",  "New Drawing",       icon, startDraw,    &model);

    loadConfig(APPLET_WIDGET(applet)->privcfgpath, &model);

    // install appropriate handlers
    disableSessionManagement(&model);
    timeout = gtk_timeout_add(250, timeoutFunc, (gpointer)&model);

    // special corba main loop
    applet_widget_gtk_main();
    return 0;
}
