/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* SAMLNameIdentifier.cpp - SAML name identifier implementation

   Scott Cantor
   4/13/04

   $History:$
*/

#include "internal.h"

using namespace saml;
using namespace std;

SAMLNameIdentifier::SAMLNameIdentifierFactoryMap SAMLNameIdentifier::m_map;

SAMLNameIdentifier* SAMLNameIdentifier::getInstance(DOMElement* e)
{
	if (!m_map.empty()) {
#ifdef HAVE_GOOD_STL
		const XMLCh* format=e ? e->getAttributeNS(NULL,L(Format)) : NULL;
#else
		auto_ptr_char temp(e ? e->getAttributeNS(NULL,L(Format)) : NULL);
		const char* format=temp.get();
#endif
		if (format) {
		    SAMLNameIdentifierFactoryMap::const_iterator i=m_map.find(format);
		    if (i!=m_map.end())
		        return (i->second)(e);
		}
	}
    return new SAMLNameIdentifier(e);
}

SAMLNameIdentifier* SAMLNameIdentifier::getInstance(istream& in)
{
    XML::Parser p;
    XML::StreamInputSource src(in);
    Wrapper4InputSource dsrc(&src,false);
    DOMDocument* doc=p.parse(dsrc);
    try {
        SAMLNameIdentifier* n=getInstance(doc->getDocumentElement());
        if (n)
            n->setDocument(doc);
        return n;
    }
    catch(...) {
        doc->release();
        throw;
    }
}

SAMLNameIdentifier::SAMLNameIdentifier(const XMLCh* name, const XMLCh* nameQualifier, const XMLCh* format)
{
    RTTI(SAMLNameIdentifier);
    m_name=XML::assign(name);
    m_nameQualifier=XML::assign(nameQualifier);
    m_format=XML::assign(format);
}

SAMLNameIdentifier::SAMLNameIdentifier(DOMElement* e) : m_name(NULL), m_nameQualifier(NULL), m_format(NULL)
{
    RTTI(SAMLNameIdentifier);
    fromDOM(e);
}

SAMLNameIdentifier::SAMLNameIdentifier(istream& in) : SAMLObject(in), m_name(NULL), m_nameQualifier(NULL), m_format(NULL)
{
    RTTI(SAMLNameIdentifier);
    fromDOM(m_document->getDocumentElement());
}

SAMLNameIdentifier::~SAMLNameIdentifier()
{
    if (m_bOwnStrings) {
        XMLString::release(&m_name);
        XMLString::release(&m_nameQualifier);
        XMLString::release(&m_format);
    }
}

void SAMLNameIdentifier::ownStrings()
{
    if (!m_bOwnStrings) {
        m_name=XML::assign(m_name);
        m_format=XML::assign(m_format);
        m_nameQualifier=XML::assign(m_nameQualifier);
        m_bOwnStrings=true;
    }
}

void SAMLNameIdentifier::fromDOM(DOMElement* e)
{
    SAMLObject::fromDOM(e);

    if (SAMLConfig::getConfig().strict_dom_checking && !XML::isElementNamed(e,XML::SAML_NS,L(NameIdentifier)))
        throw MalformedException("SAMLNameIdentifier::fromDOM() requires saml:NameIdentifier at root");

    m_nameQualifier = const_cast<XMLCh*>(e->getAttributeNS(NULL,L(NameQualifier)));
    m_format = const_cast<XMLCh*>(e->getAttributeNS(NULL,L(Format)));
    if (e->hasChildNodes())
        m_name = const_cast<XMLCh*>(e->getFirstChild()->getNodeValue());
        
    checkValidity();
}

void SAMLNameIdentifier::setName(const XMLCh* name)
{
    if (XML::isEmpty(name))
        throw SAMLException("name cannot be empty");

    if (m_bOwnStrings)
        XMLString::release(&m_name);
    else {
        m_name=NULL;
        ownStrings();
    }
    m_name=XML::assign(name);
    setDirty();
}

void SAMLNameIdentifier::setNameQualifier(const XMLCh* nq)
{
    if (m_bOwnStrings)
        XMLString::release(&m_nameQualifier);
    else {
        m_nameQualifier=NULL;
        ownStrings();
    }
    m_nameQualifier=XML::assign(nq);
    setDirty();
}

void SAMLNameIdentifier::setFormat(const XMLCh* format)
{
    if (m_bOwnStrings)
        XMLString::release(&m_format);
    else {
        m_format=NULL;
        ownStrings();
    }
    m_format=XML::assign(format);
    setDirty();
}

DOMElement* SAMLNameIdentifier::buildRoot(DOMDocument* doc, bool xmlns) const
{
    DOMElement* e = doc->createElementNS(XML::SAML_NS, L(NameIdentifier));
    if (xmlns)
        e->setAttributeNS(XML::XMLNS_NS, L(xmlns), XML::SAML_NS);
    return e;
}

DOMNode* SAMLNameIdentifier::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLObject::toDOM(doc,xmlns);
    DOMElement* nameid=static_cast<DOMElement*>(m_root);
    doc=nameid->getOwnerDocument();

    if (m_bDirty) {
        if (!XML::isEmpty(m_nameQualifier))
            nameid->setAttributeNS(NULL,L(NameQualifier), m_nameQualifier);
        if (!XML::isEmpty(m_format))
            nameid->setAttributeNS(NULL,L(Format), m_format);
        nameid->appendChild(doc->createTextNode(m_name));
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(nameid,XML::SAML_NS);
    }
    
    return m_root;
}

void SAMLNameIdentifier::checkValidity() const
{
    if (XML::isEmpty(m_name))
        throw MalformedException("NameIdentifier is invalid, requires name");
}

SAMLObject* SAMLNameIdentifier::clone() const
{
    return new SAMLNameIdentifier(m_name,m_nameQualifier,m_format);
}

const XMLCh SAMLNameIdentifier::UNSPECIFIED[] = // urn:oasis:names:tc:SAML:1.1:nameid-format:unspecified
{
    chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
    chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_1, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_i, chLatin_d, chDash,
    chLatin_f, chLatin_o, chLatin_r, chLatin_m, chLatin_a, chLatin_t, chColon,
    chLatin_u, chLatin_n, chLatin_s, chLatin_p, chLatin_e, chLatin_c, chLatin_i, chLatin_f, chLatin_i, chLatin_e, chLatin_d, chLatin_d, chNull
};

const XMLCh SAMLNameIdentifier::EMAIL[] = // urn:oasis:names:tc:SAML:1.1:nameid-format:emailAddress
{
    chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
    chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_1, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_i, chLatin_d, chDash,
    chLatin_f, chLatin_o, chLatin_r, chLatin_m, chLatin_a, chLatin_t, chColon,
    chLatin_e, chLatin_m, chLatin_a, chLatin_i, chLatin_l, chLatin_A, chLatin_d, chLatin_d, chLatin_r, chLatin_e, chLatin_s, chLatin_s, chNull
};

const XMLCh SAMLNameIdentifier::X509[] = // urn:oasis:names:tc:SAML:1.1:nameid-format:X509SubjectName
{
    chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
    chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_1, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_i, chLatin_d, chDash,
    chLatin_f, chLatin_o, chLatin_r, chLatin_m, chLatin_a, chLatin_t, chColon,
    chLatin_X, chDigit_5, chDigit_0, chDigit_9, chLatin_S, chLatin_u, chLatin_b, chLatin_j, chLatin_e, chLatin_c, chLatin_t,
    chLatin_N, chLatin_a, chLatin_m, chLatin_e, chNull
};

const XMLCh SAMLNameIdentifier::WINDOWS[] = // urn:oasis:names:tc:SAML:1.1:nameid-format:WindowsDomainQualifiedName
{
    chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
    chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_1, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_i, chLatin_d, chDash,
    chLatin_f, chLatin_o, chLatin_r, chLatin_m, chLatin_a, chLatin_t, chColon,
    chLatin_W, chLatin_i, chLatin_n, chLatin_d, chLatin_o, chLatin_w, chLatin_s,
    chLatin_D, chLatin_o, chLatin_m, chLatin_a, chLatin_i, chLatin_n,
    chLatin_Q, chLatin_u, chLatin_a, chLatin_l, chLatin_i, chLatin_f, chLatin_i, chLatin_e, chLatin_d,
    chLatin_N, chLatin_a, chLatin_m, chLatin_e, chNull
};
