/*
 *  Sarien AGI :: Copyright (C) 1999 Dark Fiber 
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "sarien.h"
#include "agi.h"
#include "font.h"
#include "words.h"
#include "objects.h"
#include "gfx.h"
#include "picture.h"
#include "sound.h"
#include "console.h"


UINT16	run_game(void);
UINT16	run_game2(void);
UINT16	detect_game(UINT8* fname);

UINT16	show_words(void);
UINT16	show_objects(void);
UINT16	view_pictures(void);

/* For the interactive picture viewer */
UINT8	show_screen_mode = 'x';

volatile UINT32 clock_ticks;
volatile UINT32 clock_count;


extern int optind;
extern UINT8 console_active;
extern UINT16 console_y;
extern UINT8 o_status;

void interpret_cycle(void);
void update_objects (void);
void calc_obj_motion(void);
void normal_motion  (UINT8 entry, SINT16 x, SINT16 y);
void adj_pos        (UINT8 entry, UINT8 x, UINT8 y);
void move_egi       (UINT8 direction);


extern AGI_LOADER agi_v2;
extern AGI_LOADER agi_v3;

int main(int argc, char *argv[])
{
	UINT16	ec;

	printf("" TITLE " v" VERSION "\n"
"A Sierra On-Line(tm) Adventure Game Interpreter.\n"
"Copyright (C) 1999 Stuart George\n"
"Portions Copyright (C) 1998 Lance Ewing\n"
"Portions Copyright (C) 1999 Claudio Matsuoka\n"
"Portions Copyright (C) 1999 Felipe Rosinha\n"
#ifndef HAVE_GETOPT_LONG
"Portions Copyright (C) 1989-1997 Free Software Foundation, Inc.\n"
#endif
"\n"
"This program is free software; you can redistribute it and/or modify it\n"
"under the terms of the GNU General Public License, version 2 or later,\n"
"as published by the the Free Software Foundation.\n" 
"\n");

	clock_enabled = __FALSE;

	ec = parse_cli (argc, argv);

	if (ec == err_OK) {
		init_machine (argc, argv);
		screen_mode = GFX_MODE;
		clock_count = 0;
		clock_ticks = 0;
	}

	while (ec == err_OK)
	{
		loader = (AGI_LOADER*)NULL;
		font = (UINT8*)font_english;

		/* FIXME -- fails when switches are specified */
		ec = argc > 1 ? detect_game((UINT8*)argv[optind]) :
			detect_game((UINT8*)".");

		if(ec != err_OK) {
			ec = err_InvalidAGIFile;
			break;
		}

		printf("AGI v%i game detected.\n", loader->version);

		switch (optGameRun) {
		case gRUN_GAME:
			if (init_video () != err_OK) {
				ec = err_Unk;
				goto bail_out;
			}
			report ("Enabling interpreter console\n");
			console_init ();
			report ("--- Starting console ---\n\n");
			init_sound ();
			break;
		case gVIEW_PICTURES:
			if (init_video () != err_OK) {
				ec = err_Unk;
				goto bail_out;
			}
			break;
		case gSHOW_WORDS:
		case gSHOW_OBJECTS:
			break;
		}

		do
		{
			ec = agi_init();

			if (ec == err_OK)
			{
				/* setup machine specific AGI flags, etc */
				setvar (V_computer, 0);	/* IBM PC */
				setvar (V_soundgen, 1);	/* IBM PC SOUND */
				setvar (V_max_input_chars, 38);
				setvar (V_monitor, 0x3); /* EGA monitor */

				horizon = HORIZON;
				control_mode = program_control;
				o_status = 5;	/* FIXME */

				ec = run_game();
			}

			/* deinit our resources */
			agi_deinit();
		} while (ec==err_RestartGame);

		switch (optGameRun)
		{
		case gRUN_GAME:
			deinit_sound ();
		case gVIEW_PICTURES:
			deinit_video ();
		case gSHOW_WORDS:
		case gSHOW_OBJECTS:
			break;
		}

		break;
	}

bail_out:
	if (ec != err_OK && ec != err_DoNothing)
	{
		printf ("Error : %04i : ", ec);
		switch (ec)
		{
		case err_BadCLISwitch:
			printf("Bad CLI switch.\n");
			break;
		case err_InvalidAGIFile:
			printf("Invalid AGI file, or no AGI file in "
				"current directory.\n");
			break;
		case err_BadFileOpen:
			printf("Unable to open file.\n");
			break;
		case err_NotEnoughMemory:
			printf("Not enough memory.\n");
			break;
		case err_BadResource:
			printf("Error in resource.\n");
			break;
		case err_UnknownAGIVersion:
			printf("Unknown AGI version.\n");
			break;
		}
		printf("\nFor extended help, use -h for syntax\n");
	}

#if 0
	/* segfault in linux */
	if(gname != NULL)
		free (gname);
#endif

	deinit_machine ();

	exit (ec);			/* this is for the memory system */
	return ec;
}


UINT16 detect_game (UINT8 *gn)
{
	UINT16	ec = err_OK;

	if (gname == NULL)
		gname = (UINT8*)strdup((char*)null_msg);

	loader = &agi_v2;
	ec = loader->detect_game (gn);

	if (ec != err_OK) {
		loader = &agi_v3;
		ec = loader->detect_game (gn);
	}

	return(ec);
}


UINT16 show_words(void)
{
	UINT16 ec=err_OK;
	UINT16 i;
	UINT16 uid, sid, lid;

	uid = sid = 0;
	lid = 0xFFFF;

	printf("  ID   Word\n");
	for (i = 0; i < num_words; i++)
	{
		if (lid == words[i].id)
			sid++;
		else
		{
			lid = words[i].id;
			uid++;
		}
		printf ("%4i - %s\n", words[i].id, words[i].word);
	}

	printf ("\n%li words in dictionary\n", num_words);
	printf ("%5i unique words\n", uid);
	printf ("%5i synonyms\n", sid);
	printf (
"\nThis is dodgy, only synonyms are id'd if they follow each other\n"
"e.g. in Space Quest I, knife, army knife, xenon army knife are all synonyms\n"
"but are not treated as such as they do not alphabetically follow each other.\n");

	return ec;
}


UINT16 show_objects(void)
{
	UINT16 ec=err_OK;
	UINT16 i;

	printf(" ID   Objects\n");
	for(i=0; i<num_objects; i++)
		printf("%3i - %s\n", (objects+i)->location, (objects+i)->name);

	printf("\n%li objects\n", num_objects);
	return ec;
}


UINT16 run_game (void)
{
	UINT16	ec = err_OK;

	switch (optGameRun)
	{
	case gRUN_GAME:
		ec = run_game2 ();
		break;
	case gVIEW_PICTURES:
		ec = view_pictures ();
		break;
	case gSHOW_WORDS:
		ec = show_words ();
		break;
	case gSHOW_OBJECTS:
		ec = show_objects ();
		break;
	}

	return ec;
}


UINT16 view_pictures (void)
{
	UINT16	ec = err_OK;
	UINT32	resnum = 0;
	UINT8	x[64];
	SINT16	i, pic, dir = 1;

	console_active = 0;
	console_y = 0;
	show_screen_mode = 'v';

	for (i = 0; ec == err_OK; i = 1)
	{
		for (pic = resnum; ; )
		{
			/* scan for resource */
			if (dir_pic[pic].offset != _EMPTY)
				break;

			pic += dir;
			if (pic < 0)
				pic = MAX_DIRS - 1;
			if(pic > MAX_DIRS - 1) {
				pic = 0;
				if (i == 0) {		/* no pics? */
					ec = 1;
					fprintf (stderr, "No pictures found\n");
					goto end_view;
				}
			}
		}
		resnum = pic;

		ec = loader->load_resource (rPICTURE, resnum);

		if (ec != err_OK)
			continue;

		sprintf ((char*)x, "Picture:%3li     [drawing]     Show: %3s",
			resnum, optShowScreenDraw ? " on" : "off");
		print_text (x, 0, 4, 190, strlen ((char*)x) + 1, 15, 0);

		/* decodes the raw data to useable form */
		decode_picture (resnum);

		switch (show_screen_mode) {
		case 'x':
			put_block_buffer (xdata_data, 0, 0, _WIDTH, _HEIGHT);
			break;
		case 'c':
			put_block_buffer (control_data, 0, 0, _WIDTH, _HEIGHT);
			break;
		case 'p':
			put_block_buffer (priority_data, 0, 0, _WIDTH, _HEIGHT);
			break;
		case 'v':
		default:
			dump_screenX ();
			break;
		}

update_statusline:
		sprintf ((char*)x, "V:Vis C:Con P:Pri X:P+C   +:Next -:Prev");
		print_text (x, 0, 4, 170, strlen ((char*)x) + 1, 15, 0);
		sprintf ((char*)x, "R:Redraw      D:Show toggle      Q:Quit");
		print_text (x, 0, 4, 180, strlen ((char*)x) + 1, 15, 0);
		sprintf ((char*)x, "Picture:%3li                   Show: %3s",
			resnum, optShowScreenDraw ? " on" : "off");
		print_text (x, 0, 4, 190, strlen ((char*)x) + 1, 15, 0);

		put_screen ();

		while (42)
    		{
    			switch (tolower (gfx->get_key() & 0xFF))
    			{
    			case 'q':
				goto end_view;
    			case 'v':
				show_screen_mode = 'v';
    				dump_screenX ();
    				break;
    			case 'p':
    			case 'z':
				show_screen_mode = 'p';
    				dump_pri (resnum);
 				break;
    			case 'c':
				show_screen_mode = 'c';
    				dump_con (resnum);
				break;
			case 'd':
				optShowScreenDraw = !optShowScreenDraw;
				goto update_statusline;
    			case 'x':
				show_screen_mode = 'x';
    				dump_x (resnum);
  				break;
			case 'r':
				goto next_pic;
    			case '+':
    				pic = resnum;
 				if (pic < MAX_DIRS - 1)
    					pic++;
    				else
    					pic = 0;
    				dir = 1;
				goto next_pic;
    			case '-':
    				pic = resnum;
    				if (pic > 0)
    					pic--;
    				else
    					pic = MAX_DIRS - 1;
    				i = 0;
    				dir = -1;
				goto next_pic;
    			}
    		}
next_pic:
    		loader->unload_resource (rPICTURE, resnum);
    		resnum = pic;
	}
end_view:
	return ec;
}

