/************************************************************************/
/* Module : search.c					                */
/* Purpose: find and replace module for mpsql     	                */
/* By     : Keith R. Davis					        */
/* Date   : 1/20/99					                */
/* Notes  : Copyright(c) 1996-99 Mutiny Bay Software			*/
/************************************************************************/

#include <Xm/Xm.h>		/* motif lib header		*/
#include <Xm/PrimitiveP.h>      /* motif primitive widget head. */
#include <Xm/PushB.h>		/* push button widget header	*/
#include <Xm/RowColumn.h>	/* row/col widget header	*/
#include <Xm/Label.h>		/* label widget header		*/
#include <Xm/TextF.h>		/* text widget header		*/
#include <Xm/MessageB.h>	/* message box header	        */
#include <string.h>		/* string lib header		*/
#include <stdio.h>              /* stdio header                 */
#include <malloc.h>             /* malloc header                */
#include <help.h>               /* help lib header              */

#include "textBuf.h"            /* text widget headers          */
#include "textSel.h"
#include "text.h"
#include "textDisp.h"
#include "textP.h"
#include "window.h"
#include "highlight.h"
#include "highlightData.h"
#include "regularExp.h"

#include "search.h"             /* search header                */
#include "msgbox.h"             /* message box header           */
#include "mquel.h"              /* mquel header                 */
#include "util.h"               /* utility func. header         */
#include "xutil.h"              /* X utils                      */

#define MAX_SEARCH_STR_SZ 512   /* max search/replace str size  */

enum SearchType {SEARCH_LITERAL, SEARCH_CASE_SENSE, SEARCH_REGEX};
enum SearchDirection {SEARCH_FORWARD, SEARCH_BACKWARD};

static char find_str[MAX_SEARCH_STR_SZ];	        /* find string 	         	        */
static char replace_find_str[MAX_SEARCH_STR_SZ];	/* replace find string 	         	*/
static char replace_with_str[MAX_SEARCH_STR_SZ];	/* replace with string 	         	*/

static int isStartOfLine(char *string, int beginPos);
static int isStartOfWord(char *string, int beginPos);
static int forwardRegexSearch(char *string, char *searchString, int wrap,
	int beginPos, int *startPos, int *endPos, char *delimiters);
static int backwardRegexSearch(char *string, char *searchString,
	int wrap, int beginPos, int *startPos, int *endPos, char *delimiters);
static int searchRegex(char *string, char *searchString, int direction,
	int wrap, int beginPos, int *startPos, int *endPos, char *delimiters);
static Boolean checkRegex(char *searchString);
static Boolean search(char *searchString, int direction);
char *ReplaceAllInString(char *inString, char *searchString,
			 char *replaceString, int *copyStart,
			 int *copyEnd, int *replacementLength, char *delimiters);
int ReplaceInSelection(char *searchString, char *replaceString);
static void replaceUsingRE(char *searchStr, char *replaceStr, char *sourceStr,
			   char *destStr, int maxDestLen, char *delimiters);

/************************************************************************/
/* Function: SEARCH_FindDlg                                             */
/* Purpose : find text dialog                                           */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void SEARCH_FindDlg(Widget w, XtPointer clientdata, XtPointer callData)
{
  const char *title = "Find";                    /* dialog title        */
  Widget rc;                                     /* rowcol widget       */
  Widget fbtn;                                   /* find btn widget     */
  static Widget find_dialog = NULL;              /* dialog widget       */

  if(!find_dialog){
    find_dialog = XmCreateMessageDialog(AppWidgetsPtr->mainwindow, "finddialog", NULL, 0);
  
    XtVaSetValues(find_dialog,
		  XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL,
		  XmNautoUnmanage, False,
		  XtVaTypedArg, XmNdialogTitle, XmRString,
		  title, strlen(title)+1,
		  NULL);

    /* remove uneeded children */
    XtUnmanageChild(XmMessageBoxGetChild(find_dialog, XmDIALOG_OK_BUTTON));
    XtUnmanageChild(XmMessageBoxGetChild(find_dialog, XmDIALOG_SYMBOL_LABEL));
    XtUnmanageChild(XmMessageBoxGetChild(find_dialog, XmDIALOG_MESSAGE_LABEL));

    fbtn = XtVaCreateManagedWidget("findbtn", xmPushButtonWidgetClass, find_dialog,
				   NULL);

    /* setup dialog */
    rc = XtVaCreateManagedWidget("rc", xmRowColumnWidgetClass, find_dialog, 
				 XmNnumColumns, 2,
				 XmNpacking, XmPACK_TIGHT,
				 XmNorientation, XmVERTICAL,
				 NULL);

    XtCreateManagedWidget("String to find (regexp):", xmLabelWidgetClass, rc, NULL, 0);
    AppWidgetsPtr->find = XtVaCreateManagedWidget("find", xmTextFieldWidgetClass,
						  rc,
						  XmNmaxLength, MAX_SEARCH_STR_SZ,
						  NULL);

    /* add callback */
    XtAddCallback(fbtn, XmNactivateCallback, SEARCH_FindOkCB, NULL);
    XtAddCallback(find_dialog, XmNhelpCallback, get_help, "mpsql_menu_search.html");
  }

  XUTIL_SetFocus(find_dialog, AppWidgetsPtr->find);

  if(find_str != NULL)
    XmTextSetString(AppWidgetsPtr->find, find_str);
 
  XtManageChild(find_dialog);
}

/************************************************************************/
/* Function: SEARCH_FindOkCB                                            */
/* Purpose : Searches for the string passed in the find dialog          */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void SEARCH_FindOkCB(Widget w, XtPointer clientdata, XtPointer callData)
{
  int direction;
  Boolean resp;

  direction = SEARCH_FORWARD;
  
  strcpy(find_str, doubleTrim((char *)XmTextGetString(AppWidgetsPtr->find))); 

  if (*find_str == '\0')
    return;

  if(checkRegex(find_str)){
    XUTIL_BeginWait(XtParent(w));
    resp = search(find_str, direction);
    XUTIL_EndWait(XtParent(w));
  }
}

/************************************************************************/
/* Function: SEARCH_FindAgain                                           */
/* Purpose : repeats search for the string passed in the find dialog    */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void SEARCH_FindAgain(Widget w, XtPointer clientdata, XtPointer callData)
{
  int direction;
  Boolean resp;

  direction = SEARCH_FORWARD;

  if (*find_str == '\0')
    return;
  
  if(checkRegex(find_str)){
    XUTIL_BeginWait(AppWidgetsPtr->mainwindow);
    resp = search(find_str, direction);
    XUTIL_EndWait(AppWidgetsPtr->mainwindow);
  }
}

/************************************************************************/
/* Function: SEARCH_ReplaceDlg                                          */
/* Purpose : replace text dialog                                        */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void SEARCH_ReplaceDlg(Widget w, XtPointer clientdata, XtPointer callData)
{
  const char *title = "Replace";                  /* dialog title             */
  Widget rc;                                      /* rowcol widget            */
  Widget find_btn;                                /* find  btn widget         */ 
  Widget replace_btn;                             /* replace btn widget       */ 
  Widget replaceall_btn;                          /* replace all btn widget   */
  static Widget replace_dialog = NULL;            /* dialog widget            */

  if(!replace_dialog){
    replace_dialog = XmCreateMessageDialog(AppWidgetsPtr->mainwindow, "replacedialog", NULL, 0);
  
    /* setup dialog */
    XtVaSetValues(replace_dialog,
		  XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL,
		  XmNautoUnmanage, False,
		  XtVaTypedArg, XmNdialogTitle, XmRString,
		  title, strlen(title)+1,
		  NULL);

    /* remove uneeded children */
    XtUnmanageChild(XmMessageBoxGetChild(replace_dialog, XmDIALOG_OK_BUTTON));
    XtUnmanageChild(XmMessageBoxGetChild(replace_dialog, XmDIALOG_SYMBOL_LABEL));
    XtUnmanageChild(XmMessageBoxGetChild(replace_dialog, XmDIALOG_MESSAGE_LABEL));
    
    find_btn = XtVaCreateManagedWidget("fbtn", xmPushButtonWidgetClass, replace_dialog,
					  NULL);
    replace_btn = XtVaCreateManagedWidget("rbtn", xmPushButtonWidgetClass, replace_dialog,
					  NULL);
    replaceall_btn = XtVaCreateManagedWidget("rallbtn", xmPushButtonWidgetClass, replace_dialog,
					     NULL);

    rc = XtVaCreateManagedWidget("rc", xmRowColumnWidgetClass, replace_dialog, 
				 XmNnumColumns, 2,
				 XmNpacking, XmPACK_TIGHT,
				 XmNorientation, XmVERTICAL,
				 NULL);

    XtCreateManagedWidget("String to find (regexp):", xmLabelWidgetClass, rc, NULL, 0);
    AppWidgetsPtr->replacefind = XtVaCreateManagedWidget("rfind", xmTextFieldWidgetClass,
							  rc,
							  XmNmaxLength, MAX_SEARCH_STR_SZ,
							  NULL);

    XtCreateManagedWidget("Replace with:", xmLabelWidgetClass, rc, NULL, 0);
    AppWidgetsPtr->replacewith = XtVaCreateManagedWidget("rwith", xmTextFieldWidgetClass,
							  rc,
							  XmNmaxLength, MAX_SEARCH_STR_SZ,
							  NULL);

    /* add callback */
    XtAddCallback(find_btn, XmNactivateCallback, SEARCH_ReplaceFindCB, NULL);
    XtAddCallback(replace_btn, XmNactivateCallback, SEARCH_ReplaceCB, NULL);
    XtAddCallback(replaceall_btn, XmNactivateCallback, SEARCH_ReplaceAllCB, NULL);
    XtAddCallback(replace_dialog, XmNhelpCallback, get_help, "mpsql_menu_search.html");
  }

  XUTIL_SetFocus(replace_dialog, AppWidgetsPtr->replacefind);

  if(replace_find_str != NULL)
    XmTextSetString(AppWidgetsPtr->replacefind, replace_find_str);

  if(replace_with_str != NULL)
    XmTextSetString(AppWidgetsPtr->replacewith, replace_with_str);
 
  XtManageChild(replace_dialog);
}

/************************************************************************/
/* Function: SEARCH_RelaceFindCB                                        */
/* Purpose : searches for the string passed in the replace dialog       */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void SEARCH_ReplaceFindCB(Widget w, XtPointer clientdata, XtPointer callData)
{
  int direction;
  Boolean resp;

  direction = SEARCH_FORWARD;
  
  strcpy(replace_find_str, doubleTrim((char *)XmTextGetString(AppWidgetsPtr->replacefind))); 

  if (*replace_find_str == '\0')
    return;
  
  if(checkRegex(replace_find_str)){
    XUTIL_BeginWait(XtParent(w));
    resp = search(replace_find_str, direction);
    XUTIL_EndWait(XtParent(w));
  }
}

/************************************************************************/
/* Function: SEARCH_RelaceCB                                            */
/* Purpose : searches/replaces the string passed in the replace dialog  */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void SEARCH_ReplaceCB(Widget w, XtPointer clientdata, XtPointer callData)
{
  int direction;
  int resp;

  direction = SEARCH_FORWARD;
  
  strcpy(replace_find_str, doubleTrim((char *)XmTextGetString(AppWidgetsPtr->replacefind))); 
  strcpy(replace_with_str, doubleTrim((char *)XmTextGetString(AppWidgetsPtr->replacewith)));

  if (*replace_find_str == '\0')
    return;
  
  resp = ReplaceInSelection(replace_find_str, replace_with_str);
}

/************************************************************************/
/* Function: SEARCH_ReplaceAllCB                                        */
/* Purpose : searches/replaces all occurances of the find string        */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void SEARCH_ReplaceAllCB(Widget w, XtPointer clientdata, XtPointer callData)
{
  int direction;
  Boolean resp;
  char *fileString, *newFileString;
  int copyStart, copyEnd, replacementLen;

  XUTIL_BeginWait(XtParent(w));

  direction = SEARCH_FORWARD;
  
  strcpy(replace_find_str, doubleTrim((char *)XmTextGetString(AppWidgetsPtr->replacefind)));
  strcpy(replace_with_str, doubleTrim((char *)XmTextGetString(AppWidgetsPtr->replacewith)));

  if (*replace_find_str == '\0'){
    XUTIL_EndWait(XtParent(w));
    return;
  }
	
  /* get the entire text buffer from the text area widget */
  fileString = BufGetAll(WinDataPtr->buffer);
    
  newFileString = ReplaceAllInString(fileString, replace_find_str, replace_with_str,
				     &copyStart, &copyEnd, &replacementLen,
				     GetWindowDelimiters(WinDataPtr));
  if (newFileString == NULL) {
    XUTIL_EndWait(XtParent(w));
    MSGBOX_Message("Replace", "String was not found", AppWidgetsPtr->mainwindow);
    return;
  }
  XtFree(fileString);
  
  /* replace the contents of the text widget with the substituted text */
  BufReplace(WinDataPtr->buffer, copyStart, copyEnd, newFileString);
  
  /* Move the cursor to the end of the last replacement */
  TextSetCursorPos(AppWidgetsPtr->sqlwindow, copyStart + replacementLen);
  
  XtFree(newFileString);
  XUTIL_EndWait(XtParent(w));
  return;
}

/************************************************************************/
/* Functions: Regex buffer search routines                              */
/* Purpose  : search the supplied string for regex patterns             */
/* Notes    :                                                           */
/************************************************************************/

/******************************************************************************/
/* Determine if beginPos begins a line or word in string.  Note that we should*/
/* be asking the caller to provide this information about the string as a     */ 
/* whole, but it probably isn't worth all the extra work                      */
/******************************************************************************/

static int isStartOfLine(char *string, int beginPos)
{
    return beginPos == 0 ? TRUE : string[beginPos-1] == '\n';
}

static int isStartOfWord(char *string, int beginPos)
{
    char *d, prevChar;
    
    if (beginPos == 0)
    	return TRUE;
    prevChar = string[beginPos-1];
    if (prevChar == ' ' || prevChar == '\t' || prevChar == '\n')
    	return TRUE;
    for (d=GetWindowDelimiters(WinDataPtr); *d!='\0'; d++)
      if (string[beginPos-1] == *d)
        return TRUE;
    return FALSE;
}

static int forwardRegexSearch(char *string, char *searchString, int wrap,
	int beginPos, int *startPos, int *endPos, char *delimiters)
{
    regexp *compiledRE = NULL;
    char *compileMsg;
    
    /* compile the search string for searching with ExecRE.  Note that
       this does not process errors from compiling the expression.  It
       assumes that the expression was checked earlier. */
    compiledRE = CompileRE(searchString, &compileMsg);
    if (compiledRE == NULL)
	return FALSE;

    /* search from beginPos to end of string */
    if (ExecRE(compiledRE, string + beginPos, NULL, FALSE,
    	    isStartOfLine(string, beginPos), isStartOfWord(string, beginPos),
    	    delimiters)) {
	*startPos = compiledRE->startp[0] - string;
	*endPos = compiledRE->endp[0] - string;
	XtFree((char *)compiledRE);
	return TRUE;
    }
    
    /* if wrap turned off, we're done */
    if (!wrap) {
    	XtFree((char *)compiledRE);
	return FALSE;
    }
    
    /* search from the beginning of the string to beginPos */
    if (ExecRE(compiledRE, string, string + beginPos, FALSE, TRUE, TRUE,
    	    delimiters)) {
	*startPos = compiledRE->startp[0] - string;
	*endPos = compiledRE->endp[0] - string;
	XtFree((char *)compiledRE);
	return TRUE;
    }

    XtFree((char *)compiledRE);
    return FALSE;
}

static int backwardRegexSearch(char *string, char *searchString,
	int wrap, int beginPos, int *startPos, int *endPos, char *delimiters)
{
    regexp *compiledRE = NULL;
    char *compileMsg;
    int length;

    /* compile the search string for searching with ExecRE */
    compiledRE = CompileRE(searchString, &compileMsg);
    if (compiledRE == NULL)
	return FALSE;

    /* search from beginPos to start of file.  A negative begin pos	*/
    /* says begin searching from the far end of the file.		*/
    if (beginPos >= 0) {
	if (ExecRE(compiledRE, string, string + beginPos, TRUE, TRUE, TRUE,
		delimiters)) {
	    *startPos = compiledRE->startp[0] - string;
	    *endPos = compiledRE->endp[0] - string;
	    XtFree((char *)compiledRE);
	    return TRUE;
	}
    }
    
    /* if wrap turned off, we're done */
    if (!wrap && beginPos >= 0) {
    	XtFree((char *)compiledRE);
    	return FALSE;
    }
    
    /* search from the end of the string to beginPos */
    if (beginPos < 0)
    	beginPos = 0;
    length = strlen(string); /* sadly, this means scanning entire string */
    if (ExecRE(compiledRE, string + beginPos, string + length, TRUE,
    	    isStartOfLine(string, beginPos), isStartOfWord(string, beginPos),
    	    delimiters)) {
	*startPos = compiledRE->startp[0] - string;
	*endPos = compiledRE->endp[0] - string;
	XtFree((char *)compiledRE);
	return TRUE;
    }
    XtFree((char *)compiledRE);
    return FALSE;
}

static int searchRegex(char *string, char *searchString, int direction,
	int wrap, int beginPos, int *startPos, int *endPos, char *delimiters)
{
    if (direction == SEARCH_FORWARD)
	return forwardRegexSearch(string, searchString, wrap, 
		beginPos, startPos, endPos, delimiters);
    else
    	return backwardRegexSearch(string, searchString, wrap, 
		beginPos, startPos, endPos, delimiters);
}

static Boolean checkRegex(char *searchString)
{
  regexp *compiledRE = NULL;
  char *compileMsg;
  char err_msg[ERR_MSG_SIZE];

  /* compile the expression and see if it is valid */
  compiledRE = CompileRE(searchString, &compileMsg);
  if (compiledRE == NULL) {
    sprintf(err_msg, "Please respecify the search string:\n%s", compileMsg);
    MSGBOX_Error("Regex Error",  err_msg,  AppWidgetsPtr->mainwindow);
    return False;
  }
  
  /* expression */
  XtFree((char *)compiledRE);
  return True;
}

static Boolean search(char *searchString, int direction)
{
  char *fileString;
  int found, fileEnd;
  int startPos, endPos;
  int beginPos, cursorPos, selStart, selEnd;
  Boolean resp;

  direction = SEARCH_FORWARD;

  selStart = -1; selEnd = -1;
  /* no selection, or no match, search relative cursor */
  cursorPos = TextGetCursorPos(AppWidgetsPtr->sqlwindow);
  if (direction == SEARCH_BACKWARD) {
    /* use the insert position - 1 for backward searches */
    beginPos = cursorPos-1;
  } else {
    /* use the insert position for forward searches */
    beginPos = cursorPos;
  }
 	
  /* get the entire text buffer from the text area widget */
  fileString = BufGetAll(WinDataPtr->buffer);
  if(strlen(fileString) == 0){
    return False;
  }
    
  /* search the string copied and present dialogs, or just beep */
  found = searchRegex(fileString, searchString, direction,
		      FALSE, beginPos, &startPos, &endPos,
		      GetWindowDelimiters(WinDataPtr));
  if (!found) {
    fileEnd = WinDataPtr->buffer->length - 1;
    if (direction == SEARCH_FORWARD && beginPos != 0) {
      resp = MSGBOX_Response("Find", 
			     "Continue search from\nbeginning of file?",
			     AppWidgetsPtr->mainwindow);
      if (!resp) {
	XtFree(fileString);
	return False;
      }
      found = searchRegex(fileString, searchString, direction,
			  FALSE, 0, &startPos, &endPos,
			  GetWindowDelimiters(WinDataPtr));
    } else if (direction == SEARCH_BACKWARD && beginPos != fileEnd) {
      resp = MSGBOX_Response("Find", 
			     "Continue search from\nend of file?",
			     AppWidgetsPtr->mainwindow);
      if (!resp) {
	XtFree(fileString);
	return False;
      }
      found = searchRegex(fileString, searchString, direction,
			  FALSE, fileEnd, &startPos, &endPos,
			  GetWindowDelimiters(WinDataPtr));
    }
    if (!found){
      MSGBOX_Message("Find",
		     "String was not found",
		     AppWidgetsPtr->mainwindow);
    }
  }

  /* if the search matched an empty string (possible with regular exps)
     beginning at the start of the search, go to the next occurrence,
     otherwise repeated finds will get "stuck" at zero-length matches */
  if (direction==SEARCH_FORWARD && beginPos==startPos && beginPos==endPos)
    if (!searchRegex(fileString, searchString, direction, FALSE,
		     fileEnd, (int*)beginPos+1, &endPos, 
		     GetWindowDelimiters(WinDataPtr))){
      XtFree(fileString);
      return False;
    }
  
  /* if matched text is already selected, just beep */
  if (selStart==startPos && selEnd==endPos) {
    XBell(XtDisplay(AppWidgetsPtr->shell), 50);
    XtFree(fileString);
    return False;
  }

  /* select the text found string */
  if(found){
    BufSelect(WinDataPtr->buffer, startPos, endPos);
    TextDMakeSelectionVisible(WinDataPtr->buffer, AppWidgetsPtr->sqlwindow);
  }

  TextSetCursorPos(AppWidgetsPtr->sqlwindow, endPos);
  
  /* free up strings */
  XtFree(fileString);
}

char *ReplaceAllInString(char *inString, char *searchString,
			 char *replaceString, int *copyStart,
			 int *copyEnd, int *replacementLength, char *delimiters)
{
    int beginPos, startPos, endPos, lastEndPos;
    int found, nFound, removeLen, replaceLen, copyLen, addLen;
    char *outString, *fillPtr;
    
    /* rehearse the search first to determine the size of the buffer needed
       to hold the substituted text.  No substitution done here yet */
    replaceLen = strlen(replaceString);
    found = TRUE;
    nFound = 0;
    removeLen = 0;
    addLen = 0;
    beginPos = 0;
    *copyStart = -1;
    while (found) {
    	found = searchRegex(inString, searchString, SEARCH_FORWARD,
    		FALSE, beginPos, &startPos, &endPos, delimiters);
	if (found) {
	    if (*copyStart < 0)
	    	*copyStart = startPos;
    	    *copyEnd = endPos;
    	    /* start next after match unless match was empty, then endPos+1 */
    	    beginPos = (startPos == endPos) ? endPos+1 : endPos;
	    nFound++;
	    removeLen += endPos - startPos;
	    {
	      char replaceResult[MAX_SEARCH_STR_SZ];
	      replaceUsingRE(searchString, replaceString, &inString[startPos],
			     replaceResult, MAX_SEARCH_STR_SZ, delimiters);
	      addLen += strlen(replaceResult);
	    }
	}
    }
    if (nFound == 0)
	return NULL;
    
    /* Allocate a new buffer to hold all of the new text between the first
       and last substitutions */
    copyLen = *copyEnd - *copyStart;
    outString = XtMalloc(copyLen - removeLen + addLen + 1);
    
    /* Scan through the text buffer again, substituting the replace string
       and copying the part between replaced text to the new buffer  */
    found = TRUE;
    beginPos = 0;
    lastEndPos = 0;
    fillPtr = outString;
    while (found) {
    	found = searchRegex(inString, searchString, SEARCH_FORWARD,
    		FALSE, beginPos, &startPos, &endPos, delimiters);
	if (found) {
	    if (beginPos != 0) {
		memcpy(fillPtr, &inString[lastEndPos], startPos - lastEndPos);
		fillPtr += startPos - lastEndPos;
	    }
	    {
    		char replaceResult[MAX_SEARCH_STR_SZ];
    		replaceUsingRE(searchString, replaceString, &inString[startPos],
    			replaceResult, MAX_SEARCH_STR_SZ, delimiters);
    		replaceLen = strlen(replaceResult);
    		memcpy(fillPtr, replaceResult, replaceLen);
	    } 
	    fillPtr += replaceLen;
	    lastEndPos = endPos;
	    /* start next after match unless match was empty, then endPos+1 */
	    beginPos = (startPos == endPos) ? endPos+1 : endPos;
	}
    }
    *fillPtr = '\0';
    *replacementLength = fillPtr - outString;
    return outString;
}

/*
** Replace all occurences of "searchString" in "window" with "replaceString"
** within the current primary selection in "window". Also adds the search and
** replace strings to the global search history.
*/
int ReplaceInSelection(char *searchString, char *replaceString)
{
    int selStart, selEnd, beginPos, startPos, endPos, realOffset, replaceLen;
    int found, anyFound, isRect, rectStart, rectEnd, lineStart, cursorPos;
    char *fileString;
    textBuffer *tempBuf;
    
    /* find out where the selection is */
    if (!BufGetSelectionPos(WinDataPtr->buffer, &selStart, &selEnd, &isRect,
    	    &rectStart, &rectEnd))
    	return FALSE;
	
    /* get the selected text */
    if (isRect) {
    	selStart = BufStartOfLine(WinDataPtr->buffer, selStart);
    	selEnd = BufEndOfLine(WinDataPtr->buffer, selEnd);
    	fileString = BufGetRange(WinDataPtr->buffer, selStart, selEnd);
    } else
    	fileString = BufGetSelectionText(WinDataPtr->buffer);
    
    /* create a temporary buffer in which to do the replacements to hide the
       intermediate steps from the display routines, and so everything can
       be undone in a single operation */
    tempBuf = BufCreate();
    BufSetAll(tempBuf, fileString);
    
    /* search the string and do the replacements in the temporary buffer */
    replaceLen = strlen(replaceString);
    found = TRUE;
    anyFound = FALSE;
    beginPos = 0;
    realOffset = 0;
    while (found) {
	found = searchRegex(fileString, searchString, SEARCH_FORWARD,
		FALSE, beginPos, &startPos, &endPos,
		GetWindowDelimiters(WinDataPtr));
	if (!found)
	    break;
	/* if the selection is rectangular, verify that the found
	   string is in the rectangle */
	if (isRect) {
	    lineStart = BufStartOfLine(tempBuf, startPos+realOffset);
	    if (BufCountDispChars(tempBuf, lineStart, startPos+realOffset) <
	    	    rectStart || BufCountDispChars(tempBuf, lineStart,
	    	    endPos+realOffset) > rectEnd) {
		beginPos = (startPos == endPos) ? endPos+1 : endPos;
		continue;
	    }
	}
	/* Make sure the match did not start past the end (regular expressions
	   can consider the artificial end of the range as the end of a line,
	   and match a fictional whole line beginning there) */
	if (startPos == selEnd - selStart) {
	    found = False;
	    break;
	}
	/* replace the string and compensate for length change */
	{
    	    char replaceResult[MAX_SEARCH_STR_SZ], *foundString;
	    foundString = BufGetRange(tempBuf, startPos+realOffset,
		    endPos+realOffset);
    	    replaceUsingRE(searchString, replaceString, foundString,
		    replaceResult, MAX_SEARCH_STR_SZ, GetWindowDelimiters(WinDataPtr));
	    XtFree(foundString);
    	    BufReplace(tempBuf, startPos+realOffset, endPos+realOffset,
    		    replaceResult);
    	    replaceLen = strlen(replaceResult);
	}
    	realOffset += replaceLen - (endPos - startPos);
    	/* start again after match unless match was empty, then endPos+1 */
    	beginPos = (startPos == endPos) ? endPos+1 : endPos;
    	cursorPos = endPos;
	anyFound = TRUE;
    }
    XtFree(fileString);
    
    /* if nothing was found, tell user and return */
    if (!anyFound) {
      MSGBOX_Message("Replace", "String was not found", AppWidgetsPtr->mainwindow);
      BufFree(tempBuf);
      return FALSE;
    }
    
    /* replace the selected range in the real buffer */
    fileString = BufGetAll(tempBuf);
    BufFree(tempBuf);
    BufReplace(WinDataPtr->buffer, selStart, selEnd, fileString);
    XtFree(fileString);
    
    /* set the insert point at the end of the last replacement */
    TextSetCursorPos(AppWidgetsPtr->sqlwindow, selStart + cursorPos + realOffset);
    
    return TRUE;
}

/*
** Substitutes a replace string for a string that was matched using a
** regular expression.  This was added later and is very inneficient
** because instead of using the compiled regular expression that was used
** to make the match in the first place, it re-compiles the expression
** and redoes the search on the already-matched string.  This allows the
** code to continue using strings to represent the search and replace
** items.
*/  

static void replaceUsingRE(char *searchStr, char *replaceStr, char *sourceStr,
			   char *destStr, int maxDestLen, char *delimiters)
{
  regexp *compiledRE;
  char *compileMsg;
  
  compiledRE = CompileRE(searchStr, &compileMsg);
  ExecRE(compiledRE, sourceStr, NULL, False, True, True, delimiters);
  SubstituteRE(compiledRE, replaceStr, destStr, maxDestLen);
  XtFree((char *)compiledRE);
}
