// $Id: Cxx.cc,v 1.76 2002/02/25 14:58:11 christof Exp $
/*  glade--: C++ frontend for glade (Gtk+ User Interface Builder)
 *  Copyright (C) 1998  Christof Petig
 *  Copyright (C) 1999-2000 Adolf Petig GmbH & Co. KG, written by Christof Petig
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "Cxx.hh"
#include <strstream>
#include "Configuration.hh"
#include "Widget_type.hh"
#include "WidgetMap.hh"

// for some accesses
#include <time.h>
#include <pwd.h>
#include <sys/utsname.h>
#include <unistd.h>
#include <sys/stat.h>

#define GLADEMM_NAME PACKAGE " V" VERSION

WidgetMap_t WidgetMap;

Cxx::Cxx(Tag *tp) 
	:top(tp), current_set(0)
{}

void Cxx::CreateSupportFiles()
{  if (Configuration.lookup_table)
   {  CxxFile support_h(FileNameByType(File_SUPPORT_HH));
      WriteHeader(support_h,File_SUPPORT_HH);
      CxxFile support_c(FileNameByType(File_SUPPORT_CC)); 
      WriteHeader(support_c,File_SUPPORT_CC);
      
      support_h.Include("string");
      support_h.Include("map");
      
      support_c.Include("glademm_support"+Configuration.header_suffix,true);
      
   if (Configuration.lookup_table)
   {  support_h.Include("gtk--/widget.h");
      
      support_h.Declaration().Funct_ReturnType(WriterBase::GtkPrefix()+"Widget *")
      		.FunctionName("glademm_get_Widget")
      		.FunctionArg("const __STD::string &name");
      support_h.Declaration().Funct_ReturnType("void")
      		.FunctionName("glademm_set_Widget")
      		.FunctionArg("const __STD::string &name")
      		.FunctionArg(WriterBase::GtkPrefix()+"Widget *w");
      support_h.Definition().Funct_ReturnType("template <class T> T *")
      		.FunctionName("glademm_get")
      		.FunctionArg("const __STD::string &name");
      support_h.StartBlock();
      support_h.Statement("return reinterpret_cast<T*>(glademm_get_Widget(name))");
      support_h.ShortComment("2do: testen ... geht aber nicht ohne Methode gleichen Namens");
      support_h.EndBlock();

      support_c.Declaration("static std::map<const __STD::string,"+WriterBase::GtkPrefix()+"Widget *> glademm_widgets");

      support_c.Definition().Funct_ReturnType(WriterBase::GtkPrefix()+"Widget *")
      		.FunctionName("glademm_get_Widget")
      		.FunctionArg("const __STD::string &name");
      support_c.StartBlock();
      support_c.Declaration("std::map<const __STD::string,"+WriterBase::GtkPrefix()+"Widget *>::iterator i=glademm_widgets.find(name)");
      support_c.Statement("if (i==glademm_widgets.end()) return 0");
      support_c.Statement("return (*i).second");
      support_c.EndBlock();
      
      support_c.Definition().Funct_ReturnType("void")
      		.FunctionName("glademm_set_Widget")
      		.FunctionArg("const __STD::string &name")
      		.FunctionArg(WriterBase::GtkPrefix()+"Widget *val");
      support_c.StartBlock();
      support_c.Statement("glademm_widgets[name]=val");
      support_c.EndBlock();
   }
     if (Configuration.no_autoconf)
     {  makefile.Dependancy("glademm_support.o");
     }
     else 
     {  makefile_am.Dependancy(Configuration.FileName("foo",File_SUPPORT_CC,File_NODIR));
        Remember<MakeFile>(makefile_am)
        	.Dependancy(Configuration.FileName("foo",File_SUPPORT_HH,File_NODIR));
     }
      WriteFooter(support_h,File_SUPPORT_HH);
      support_h.close();
      WriteFooter(support_c,File_SUPPORT_CC);
      support_c.close();
   }
}

void Cxx::WriteTags()
{  MakeFile toplev_Make_am;
   SystemFile readme,authors,changelog,news,todo;
   SystemFile po_cl;
   std::string autogen_name;

   if (Configuration.debug) top->debug();
   if (!Configuration.bare_bones)
   {  main.open(FileNameByType(File_MAIN_CC)); WriteHeader(main,File_MAIN_CC);
      std::string fname(FileNameByType(File_MAKEFILE));
      if (Configuration.no_autoconf || access(fname.c_str(),F_OK))
      {  makefile.open(fname); WriteHeader(makefile,File_MAKEFILE);
      }
      if (!Configuration.no_autoconf)
      {  autogen_name=FileNameByType(File_AUTOGEN_SH);
         if (access(autogen_name.c_str(),F_OK))
         {  autogen.open(autogen_name,std::ios::out); 
      	    WriteHeader(autogen,File_AUTOGEN_SH);
      	 }
      	 
         makefile_am.open(FileNameByType(File_MAKEFILE_AM)); 
         WriteHeader(makefile_am,File_MAKEFILE_AM);
         
         fname=FileNameByType(File_CONFIGURE_IN);
         if (access(fname.c_str(),F_OK))
         {  configure_in.open(fname);
            WriteHeader(configure_in,File_CONFIGURE_IN);
         }
         fname=FileNameByType(File_toplevel_MAKEFILE_AM);
         if (access(fname.c_str(),F_OK))
         {  toplev_Make_am.open(fname);
            WriteHeader(toplev_Make_am,File_toplevel_MAKEFILE_AM);
         }
         fname=FileNameByType(File_README);
         if (access(fname.c_str(),F_OK))
         {  readme.open(fname);
            WriteHeader(readme,File_README);
         }
         fname=FileNameByType(File_AUTHORS);
         if (access(fname.c_str(),F_OK))
         {  authors.open(fname);
            WriteHeader(authors,File_AUTHORS);
         }
         
         changelog.open(FileNameByType(File_ChangeLog,0),std::ios::app);
         WriteHeader(changelog,File_ChangeLog);
         
         fname=FileNameByType(File_NEWS);
         if (access(fname.c_str(),F_OK))
         {  news.open(fname);
            WriteHeader(news,File_NEWS);
         }
         fname=FileNameByType(File_TODO);
         if (access(fname.c_str(),F_OK))
         {  todo.open(fname);
            WriteHeader(todo,File_TODO);
         }
         
         fname=FileNameByType(File_ACCONFIG_H);
         if (access(fname.c_str(),F_OK))
         {  acconfig.open(fname);
            WriteHeader(acconfig,File_ACCONFIG_H);
         }
         
         if (Configuration.gettext_support)
         {  po_cl.open(FileNameByType(File_po_ChangeLog,0),std::ios::app);
            WriteHeader(po_cl,File_po_ChangeLog);
            
            potfiles.open(FileNameByType(File_po_POTFILES_in)); 
            WriteHeader(potfiles,File_po_POTFILES_in);
         }
      }
   }
   
   WriteTags(*top);
   CreateSupportFiles();

   if (!Configuration.bare_bones)
   {  WriteFooter(makefile,File_MAKEFILE); makefile.close();
      WriteFooter(main,File_MAIN_CC); main.close();
      if (!Configuration.no_autoconf)
      {  WriteFooter(makefile_am,File_MAKEFILE_AM); makefile_am.close();
         WriteFooter(autogen,File_AUTOGEN_SH); autogen.close();
         chmod(autogen_name.c_str(),0755);
         WriteFooter(configure_in,File_CONFIGURE_IN); configure_in.close();
         WriteFooter(toplev_Make_am,File_toplevel_MAKEFILE_AM);
         			toplev_Make_am.close(); 
         WriteFooter(readme,File_README); readme.close();
         WriteFooter(authors,File_AUTHORS); authors.close();
         WriteFooter(changelog,File_ChangeLog); changelog.close();
         WriteFooter(news,File_NEWS); news.close();
         WriteFooter(todo,File_TODO); todo.close();
         WriteFooter(acconfig,File_TODO); acconfig.close();
         
         if (Configuration.gettext_support)
         {  WriteFooter(po_cl,File_po_ChangeLog); po_cl.close();
            WriteFooter(potfiles,File_po_POTFILES_in); potfiles.close();
         }
      }
   }
}

void Cxx::WriteTags(Tag &t)
{  // 2do: make names unique?
   if (t.Type()!="widget")
   {  for(Tag::iterator i=t.begin();i!=t.end();++i) 
         WriteTags(*i);
   }
   else
   {  ApplyPreferences(t); // XXX better scanning!
//      if (Configuration.debug) w.debug();
      Cxx_Fileset fs(t,*this);
      fs.WriteFiles(true);
   }
}

// XXX: rename widgets to get uniqueness if necessary
void Cxx::ApplyPreferences(Tag &t,int level)
{  Widget w(&t); // we need pointer to get a changeable widget

   WidgetMap[w.Name()]=&t;
   if (!level)
   {  bool seperate_class=t.getBool(CXX_SEPERATE_CLASS),
   	   seperate_file=t.getBool(CXX_SEPERATE_FILE);
      if (!seperate_class || !seperate_file)
      {  // std::cerr << w.Name() << ": toplevel widgets have their own class & file, marked\n";
         if (!seperate_class)
         {  w.mark(CXX_SEPERATE_CLASS);
            seperate_class=true;
         }
         if (!seperate_file)
         {  w.mark(CXX_SEPERATE_FILE);
            seperate_file=true;
         }
      }
   }
   LookupWriter(w).ApplyPreferences(t);
   if (level && t.getBool(CXX_SEPERATE_FILE) && !t.getBool(CXX_SEPERATE_CLASS))
   {  std::cerr << w.Name() << "Code error (" << w.Class() << ")\n";
   }
  
   // recursively apply
   for (Widget::iterator i=w.begin();i!=w.end();++i)
         ApplyPreferences(i.get_Tag(),level+1);
}

void Cxx::WriteHeader(CxxFile &f,File_type tp)
{  switch(tp)
   {  
      case File_SUPPORT_CC:
         WriteCreation(f,tp);
      	 f << "//\n"
	      "// some glademm convenience functions\n"
	      "//\n"
	      "// DO NOT EDIT THIS FILE !\n\n";
	 f.Include("config.h",true); // always a good idea
         break;
      case File_SUPPORT_HH:
         WriteCreation(f,tp);
      	 f << "//\n"
	      "// some glademm convenience functions\n"
	      "//\n"
	      "// DO NOT EDIT THIS FILE !\n\n"
	      "#ifndef GLADEMM_SUPPORT_HH\n"
	      "#define GLADEMM_SUPPORT_HH\n";
         break;
      case File_MAIN_CC:
         WriteCreation(f,tp);
   	 f << "//\n// newer (non customized) versions of this file go to ";
   	 f << Configuration.FileName("",tp,File_GLADE|File_NODIR) << "\n"
   	 	"\n"
		"// This file is for your program, I won't touch it again!\n\n";
	 if (!Configuration.gnome_support)
	    f.Include("gtk--/main.h");
	 else 
	 {  f.Include("config.h");
	    f.Include("gnome--/main.h");
	    if (Configuration.gettext_support)
	       f.Include("libgnome/gnome-i18n.h");
	 }
   	 f << '\n';
   	 break;
      case File_ACCONFIG_H:
         WriteCreation(f,tp);
   	 f <<   "\n"
	        "/* acconfig.h\n"
	        " * This file is in the public domain.\n"
	        " * \n" // it didn't like empty lines
		" * Descriptive text for the C preprocessor macros that\n"
		" * the distributed Autoconf macros can define.\n"
		" * These entries are sometimes used by macros\n"
		" * which glade-- uses.\n"
		" */\n";
	 if (Configuration.gettext_support)
      	    f <<"#undef ENABLE_NLS\n"
		"#undef HAVE_CATGETS\n"
		"#undef HAVE_GETTEXT\n"
		"#undef HAVE_LC_MESSAGES\n"
		"#undef HAVE_STPCPY\n"
		"#undef HAVE_LIBSM\n"
		"#undef PACKAGE_LOCALE_DIR\n";
	 // CHECKME: is this correct?
	 if (Configuration.gnome_support)
	    f <<"#undef PACKAGE_DATA_DIR\n"
		"#undef PACKAGE_SOURCE_DIR\n";
	 break;
      default: assert(0);
   }	
}

void Cxx::WriteHeader(SystemFile &f,File_type tp)
{  if (!f.good()) return;
   switch(tp)
   {  
      case File_MAKEFILE:
         WriteCreation(f,tp);
   	 f << "#\n# newer (non customized) versions of this file go to ";
   	 f << Configuration.FileName("",tp,File_GLADE|File_NODIR) << '\n';
   	 if (Configuration.no_autoconf)
   	 {  std::string ENABLE_NLS;
   	    if (Configuration.gettext_support) ENABLE_NLS="-DENABLE_NLS ";
   	    if (Configuration.use_libglade)
   	    {  std::string gnome(Configuration.gnome_support?"gnome":"");
   	       std::string lgnomemm(Configuration.gnome_support?"-lgnomemm":"");
   	       f << "\n"
   	         "CXXFLAGS=$(CFLAGS) "<< ENABLE_NLS <<"$(shell gtkmm-config --cflags) $(shell libglade-config --cflags "<<gnome<<")\n"
                 "LIBS=$(shell echo $$LIBS) $(shell gtkmm-config --libs) "<<lgnomemm<<" $(shell libtool-config --libs "<<gnome<<')';
            }
   	    else if (Configuration.gnome_support)
   	       f << "\n"
                 "CXXFLAGS=$(CFLAGS) "<< ENABLE_NLS <<"$(shell gtkmm-config --cflags) $(shell gnome-config --cflags gnomemm) $(shell gnome-config --cflags gnomeui)\n"
                 "LIBS=$(shell echo $$LIBS) $(shell gtkmm-config --libs) -lgnomemm $(shell gnome-config --libs gnomemm)  $(shell gnome-config --libs gnomeui)";
   	    else
   	       f << "\n"
                 "CXXFLAGS=$(CFLAGS) $(shell gtkmm-config --cflags)\n"
                 "LIBS=$(shell echo $$LIBS) $(shell gtkmm-config --libs)";
            f << "\n"
                 "# calculate dependancies\n"
                 "\n"
                 "all: dep_all\n"
                 "\n"
                 "ifeq (.depend,$(wildcard .depend))\n"
                 "include .depend\n"
                 "dep_all: " << Configuration.main_filename << "\n"
                 "else\n"
                 "dep_all: .depend\n"
                 "\tmake all\n"
                 "endif\n"
                 "\n"
                 ".depend depend dep:\n"
                 "\tfor i in *.cc ; do $(CXX) $(CXXFLAGS) -M $$i; done > .depend\n"
                 "\n" << Configuration.main_filename << ": " 
                 << FileName("",File_MAIN_CC,File_NODIR|File_OBJECT) 
                 << ' ';
   	 }
   	 else
   	 {  f << "# this file will overwrite itself !\n"
   	 	 "\n"
   	 	 "all:\n"
   	 	 "\t./autogen.sh\n";
   	 }
         break;
      case File_MAKEFILE_AM: // in the src directory
         WriteCreation(f,tp);
         if (Configuration.use_libglade)
         {  std::cerr << "glade-- can't output a Makefile.am for libglade, yet\n"; }
   	 f << "#\n# newer (non customized) versions of this file go to ";
   	 f << FileName("",tp,File_GLADE|File_NODIR) << "\n"
   	      "bin_PROGRAMS = "<< Configuration.main_filename << "\n\n";
         f << AMName(Configuration.main_filename) << "_SOURCES = \\\n"
              "\t" << FileName("",File_MAIN_CC,File_NODIR);
         {  char buf[1024]; 
            std::ostrstream o(buf,sizeof buf);
            
   	    o << "noinst_HEADERS = " << char(0);
            o.flush();
            f.remember(buf);
         }
         break;
      case File_CONFIGURE_IN:
      	WriteCreation(f,tp);
   	   f << "\n"
		"AC_INIT("<< (Configuration.source_directory.c_str()+1)
		<< '/' << Configuration.main_filename <<".cc)\n"
		"AM_INIT_AUTOMAKE("<< Configuration.main_filename <<", 0.0)\n"
	        "AM_CONFIG_HEADER(config.h)\n"
	        "\n";
	if (Configuration.gnome_support)
	   f << "dnl Pick up the Gnome macros.\n"
	   	"AM_ACLOCAL_INCLUDE(macros)\n"
	   	"AM_MAINTAINER_MODE\n" // don't ask me why it needs this
	   	"\n";
	f <<    "AC_ISC_POSIX\n"
		"AC_PROG_CC\n"
		"AM_PROG_CC_STDC\n"
		"AC_HEADER_STDC\n"
		"AC_PROG_CPP\n"
		"AC_PROG_CXX\n"
		"AC_PROG_CXXCPP\n"
		"AM_PROG_LIBTOOL\n"
		"\n";
	if (Configuration.gnome_support)
	   f << "# GNOME--:\n"
	   	"# (These macros are in the 'macros' directory)\n"
	   	"# GNOME_INIT sets the GNOME_CONFIG variable, among other things:\n"
	   	"GNOME_INIT\n"
	   	"GNOME_COMMON_INIT\n"
	   	"GNOME_COMPILE_WARNINGS\n";
	if (Configuration.gettext_support)
	   f << "dnl Add the languages which your application supports here.\n"
	   	"ALL_LINGUAS=\"\"\n"
	   	"AM_"<<(!Configuration.gnome_support?"GNU":"GNOME")<<"_GETTEXT\n"
	   	"\n";
        if (Configuration.gettext_support) // && !Configuration.gnome_support)
		// is this correct?	
             f << "dnl Set PACKAGE_LOCALE_DIR in config.h.\n"
   		"if test \"x${prefix}\" = \"xNONE\"; then\n"
   		"  AC_DEFINE_UNQUOTED(PACKAGE_LOCALE_DIR, \"${ac_default_prefix}/${DATADIRNAME}/locale\")\n"
   		"else\n"
   		"  AC_DEFINE_UNQUOTED(PACKAGE_LOCALE_DIR, \"${prefix}/${DATADIRNAME}/locale\")\n"
   		"fi\n"
		"\n";
	   f << "AC_LANG_CPLUSPLUS\n"
		"\n";
	if (!Configuration.gnome_support) // we trust ...
	   f << "AM_PATH_GTKMM("<<Configuration.gtkmm_version.mymajor <<'.'
	  << Configuration.gtkmm_version.myminor<<'.'
	  << Configuration.gtkmm_version.mymicro<<",,\n"
	     "        AC_MSG_ERROR(Cannot find GTK--: Please install Version "
	  << Configuration.gtkmm_version.mymajor <<'.'
	  << Configuration.gtkmm_version.myminor<<'.'
	  << Configuration.gtkmm_version.mymicro<<"+))\n";
	if (Configuration.gnome_support)
	   f << "# GNOME-CONFIG script knows about gnomemm:\n"
		   "# ('gnome-config' is installed by GNOME)\n"
		   "GNOMEMM_CFLAGS=\"`$GNOME_CONFIG --cflags gnomemm`\" \n"
		   "GNOMEMM_LIBS=\"`$GNOME_CONFIG --libs gnomemm`\"\n"
		   "AC_SUBST(GNOMEMM_CFLAGS)\n"
		   "AC_SUBST(GNOMEMM_LIBS)\n";
		// perhaps separate into GNOMEMM_LIBDIR and GNOMEMM_LIBS ?
		// we should really test gnomemm!!!
	 f << "\n";
#if 0
		"dnl Subst PACKAGE_PIXMAPS_DIR.\n"
		"PACKAGE_PIXMAPS_DIR=\"`gnome-config --datadir`/pixmaps/${PACKAGE}\"\n"
		"AC_SUBST(PACKAGE_PIXMAPS_DIR)\n"
#endif
	 f << "AC_OUTPUT(Makefile " << (Configuration.source_directory.c_str()+1) << "/Makefile ";
	 if (Configuration.gettext_support) // && !Configuration.gnome_support)
	 	// glade does this too ...
	    f << "intl/Makefile ";
	 if (Configuration.gettext_support)
	    f << " po/Makefile.in ";
	 if (Configuration.gnome_support)
	    f << " macros/Makefile";
	 f << ")\n";
         break;
      case File_AUTOGEN_SH:
	if (Configuration.gnome_support)
	{  // this is a verbatim copy of gnome-skel 
	   f << "#!/bin/sh\n"
                "# Run this to generate all the initial makefiles, etc.\n";
           WriteCreation(f,tp);
	   f << "# I didn't want to put a copy of 'macros' in every generated package\n"
		"# so I try to find them at autogen.sh time and copy them here.\n"
		"# (Normally if you have access to a cvs repository a copy of macros is\n"
		"# put into your directory at checkout time. E.g. cvs.gnome.org/gnome-common)\n"
	        "if [ ! -e macros ]\n"
	        "then\n"
		"  GLADE_MACROS=`which glade | sed -e 's-bin/glade-share/glade-'`\n"
		"  if [ -r $GLADE_MACROS/gnome/gnome.m4 ]\n"
		"  then\n"
		// IIRC: this behaviour is standard for older cp, they
		//       do not accept this flag
		// on the other hand a recent gnome (debian-sid) has
		// symbolic _relative_ links (deadly when copied)
		"    if cp --dereference /dev/null /dev/zero\n"
		"    then\n"
		"      cp -r --dereference $GLADE_MACROS/gnome macros\n"
		"    else\n"
		"      cp -r $GLADE_MACROS/gnome macros\n"
		"    fi\n"
                "  else\n"
                "    echo \"I can't find glade's gnome m4 macros. Please copy them to ./macros and retry.\"\n"
                "    exit 2\n"
                "  fi\n"
                "fi\n";
           f << "\n"
                "srcdir=`dirname $0`\n"
                "test -z \"$srcdir\" && srcdir=.\n"
                "\n"
                // FIXME: package name?
                "PKG_NAME=\"" << Configuration.main_filename << "\"\n"
                "\n"
                "(test -f $srcdir/configure.in \\\n"
                "## put other tests here\n"
                ") || {\n"
                "    echo -n \"**Error**: Directory \"\\`$srcdir\\'\" does not look like the\"\n"
                "    echo \" top-level $PKG_NAME directory\"\n"
                "    exit 1\n"
                "}\n"
                "\n"
                ". $srcdir/macros/autogen.sh\n";
//               "ACLOCAL_FLAGS=\"$ACLOCAL_FLAGS -I `aclocal --print-ac-dir`/gnome\" . `aclocal --print-ac-dir`/gnome/autogen.sh\n";
        }
	else
	{f << "#!/bin/sh\n";
         WriteCreation(f,tp);
   	 f << "\n"
   	        "if test ! -f install-sh ; then touch install-sh ; fi\n"
   	        "\n"
		"MAKE=`which gnumake`\n"
		"if test ! -x \"$MAKE\" ; then MAKE=`which gmake` ; fi\n"
		"if test ! -x \"$MAKE\" ; then MAKE=`which make` ; fi\n"
		"HAVE_GNU_MAKE=`$MAKE --version|grep -c \"Free Software Foundation\"`\n"
		"\n"
		"if test \"$HAVE_GNU_MAKE\" != \"1\"; then \n"
		"echo Using non GNU Make at $MAKE\n"
		"else\n"
		"echo Found GNU Make at $MAKE ... good.\n"
		"fi\n"
		"\n"
		"echo This script runs configure and make...\n"
		"echo You did remember necessary arguments for configure, right?\n"
		"\n"
		"if test ! -x `which aclocal`  \n"
		"then echo you need autoconfig and automake to generate the Makefile\n"
		"fi\n"
		"if test ! -x `which automake`  \n"
		"then echo you need automake to generate the Makefile\n"
		"fi\n"
		"\n"
		"libtoolize --force --copy\n"
		"autoheader\n";
	   f << "aclocal\n";
	   f << "automake --add-missing --copy --gnu\n"
		"autoconf\n"
		"./configure $* && $MAKE\n";
	}
         break;
      case File_toplevel_MAKEFILE_AM:
        WriteCreation(f,tp);
        if (Configuration.gnome_support)
        {  f << "AUTOMAKE_OPTIONS = 1.4\n"
              "\n";
#if 0 // doesn't work for me (wrong path)
              "if MAINTAINER_MODE\n"
              "include $(GNOME_ACLOCAL_DIR)/gnome-macros.dep\n"
              "endif\n"
              "\n";
#endif
        }
        f << "\nSUBDIRS = "; 
        if (Configuration.gettext_support) 
           f << "po "; // does gnome need intl? glade adds it ...
        if (Configuration.gnome_support)
           f << "macros ";
        if (Configuration.gettext_support) 
           f << "intl "; // does gnome really need intl?
        f << (Configuration.source_directory.c_str()+1) << ' ';
        f << "\n"
          "\n"
          "EXTRA_DIST = AUTHORS TODO README configure ";
//        if (Configuration.gnome_support)
//           f << "\\\n""\tmacros/* ";
        f << "\n";
        break;
      case File_README:
        WriteCreation(f,tp);
        f << "\nThis project was generated using glade--\n"
        	"and the README file has not been edited since then ;-).\n";
        break;
      case File_AUTHORS:
      {
        struct passwd *pwd=getpwuid(getuid());
        struct utsname uts;
        if (uname(&uts)) 
        {  strcpy(uts.nodename,"(unknown)"); 
#ifdef HAS_DOMAINNAME   
           strcpy(uts.domainname,"(unknown)");
#endif      
        }
#ifdef HAS_REAL_NAME
	f << (pwd?pwd->HAS_REAL_NAME:"") << " <";
#endif        
        f << (pwd?pwd->pw_name:"(unknown)") << '@'
          << uts.nodename;
#ifdef HAS_DOMAINNAME     
        f << '.' << uts.domainname ;
#endif     
#ifdef HAS_REAL_NAME
	f << '>';
#endif
        f << '\n';
        break;
      }
      case File_ChangeLog:
      case File_po_ChangeLog:
      { f << '\n';
        time_t t=time(0);
        struct tm *tm=localtime(&t);
        f << tm->tm_year+1900 << '/' << tm->tm_mon+1 << '/' << tm->tm_mday;
        f << ' '<< tm->tm_hour << ':';
        char oldfill=f.OStream().fill('0');
        int oldwidth=f.OStream().width(2);
        f << tm->tm_min << ':';
        f.OStream().fill('0');
        f.OStream().width(2);
        f << tm->tm_sec << ' ' << tzname[tm->tm_isdst];
        f.OStream().fill(oldfill); 
        f.OStream().width(oldwidth);
        f << "\n\tupdate files from .glade file\n";
        break;
      }
      case File_NEWS: break;
      case File_TODO:
        f << "- fill README file with something useful\n";
        break;
      case File_po_POTFILES_in:
        f << "# List of source files containing translatable strings.\n";
        f << FileNameByType(File_MAIN_CC,0) << '\n';
        break;
      default: assert(0);
   }	
}

void Cxx::WriteFooter(CxxFile &f,File_type tp)
{  if (!f.good()) return;
   switch(tp)
   {  case File_FOO_HH:
      case File_FOO_GLADE_HH: assert(0);
         break;
      case File_MAIN_CC:
         f.Definition().Funct_ReturnType("int").FunctionName("main")
         	.FunctionArg("int argc").FunctionArg("char **argv");
         f.StartBlock();
         if (Configuration.gettext_support)
         {  f.CppIf("defined(ENABLE_NLS)");
            f.Statement("bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR)");
            f.Statement("textdomain (PACKAGE)");
            f.EndIf("ENABLE_NLS");
         }
	 if (!Configuration.gnome_support)
	    f.Declaration(WriterBase::GtkPrefix()+"Main m(&argc, &argv)");
	 else
	    f.Declaration() << WriterBase::GnomePrefix()+"Main m("
	       << "PACKAGE, VERSION, argc, argv)";
	 if (Configuration.use_libglade)
	 {  std::string gnome(Configuration.gnome_support?"_gnome":"");
	    f.Statement() << "glade"<<gnome<<"_init()";
	 }
	 f.EndLine();
         f.write_remembered(0);
      	 f.Statement("m.run()").EndLine();
      	 f.write_remembered(1);     
      	 f.Statement("return 0");
      	 f.EndBlock();
         break;
      case File_SUPPORT_HH:
	 f << "#endif /* GLADEMM_SUPPORT_HH */\n";
         break;
      default: 
         break;
   }
}

void Cxx::WriteFooter(SystemFile &f,File_type tp)
{  if (!f.good()) return;
   switch(tp)
   {  case File_FOO_HH:
      case File_MAIN_CC:
      case File_SUPPORT_HH:
      case File_ACCONFIG_H:
      case File_FOO_GLADE_HH: assert(0);
         break;
      case File_MAKEFILE_AM:
         f << "\n\n";
      	 f.write_remembered();
      	 f << "\n\n"
      	      "CXXFLAGS = @CXXFLAGS@ ";
      	 if (Configuration.gnome_support)
      	    f << "@GNOMEMM_CFLAGS@\n";
      	 else
      	    f << "@GTKMM_CFLAGS@\n";
      	 if (Configuration.gnome_support)
      	    f << "LDFLAGS= @LDFLAGS@ \n";
      	    
      	 f << '\n';
      	 f << AMName(Configuration.main_filename) << "_LDADD = @LIBS@ ";
      	 if (Configuration.gnome_support)
      	    f << "@GNOMEMM_LIBS@";
      	 else
      	    f << "@GTKMM_LIBS@";
      	 f << '\n';
         break;
      case File_MAKEFILE:
         if (Configuration.no_autoconf)
            f << "\n\t$(CXX) $(LFLAGS) -o " << Configuration.main_filename << " $^ $(LIBS)\n"
	         "\n"
	         "clean:\n"
	         "\trm -f .depend " << Configuration.main_filename << " *.o *~\n";
	 break;
      default: 
         break;
   }
}

void Cxx::WriteCreation(SystemFile &f,File_type tp)
{  if (!f.good()) return;
   time_t t=time(0);
   struct tm *tm=localtime(&t);
   const char *comment=(tp>=File_NUM_CXX_FILES)?"#":"//";
   f << comment << " generated ";
   f << tm->tm_year+1900 << '/' << tm->tm_mon+1 << '/' << tm->tm_mday;
   f << ' '<< tm->tm_hour << ':';
   char oldfill=f.OStream().fill('0');
   int oldwidth=f.OStream().width(2);
   f << tm->tm_min << ':';
   f.OStream().fill('0');
   f.OStream().width(2);
   f << tm->tm_sec << ' ' << tzname[tm->tm_isdst];
   f.OStream().fill(oldfill); f.OStream().width(oldwidth);
   struct passwd *pwd=getpwuid(getuid());
   struct utsname uts;
   if (uname(&uts)) 
   {  strcpy(uts.nodename,"(unknown)"); 
#ifdef HAS_DOMAINNAME   
      strcpy(uts.domainname,"(unknown)");
#endif      
   }
   f << " by " << (pwd?pwd->pw_name:"(unknown)") << '@'
     << uts.nodename 
#ifdef HAS_DOMAINNAME     
     << '.' << uts.domainname 
#endif     
     << '\n';
   f << comment << " using "GLADEMM_NAME"\n";
}


/*
   REMEMBERED LINES:
   
   following files use remember() for their purposes:
   
   - makefile_am:	list of header files
   - main: 0		creation of toplevel components
   - main: 1		deletion of heap objects
*/
  
const std::string Cxx::FileNameByType(const File_type &tp,int flags)
{  return Configuration.FileName("foo",tp,flags);
}
