// --------------------------------------------------------------------
// ipetoipe
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipeq.h"
#include "ipemodel.h"
#include "ipeprefs.h"

#include <QDir>

#include <cstdio>
#include <cstdlib>

#ifdef WIN32
#include <windows.h>
#endif

#ifdef WIN32
static void ipeDebugImpl(const char *msg)
{
  OutputDebugStringA(msg);
}
#else
static void ipeDebugImpl(const char *)
{
  // fprintf(stderr, "%s\n", msg);
}
#endif

static int topdforps(IpeDocument *doc, const char *src, const char *dst,
		     IpeDocument::TFormat fm, uint flags,
		     int fromPage = -1, int toPage = -1)
{
  if (fm == IpeDocument::EEps && doc->totalViews() > 1) {
    fprintf(stderr,
	    "The document contains %d views, cannot convert to EPS format.\n",
	    doc->totalViews());
    return 1;
  }

  fprintf(stderr, "Document %s has %d pages (views)\n",
	  src, doc->totalViews());

  int exitCode = IpeModel::RunLatex(doc);

  if (!exitCode) {
    if (fm == IpeDocument::EEps && doc->HasTrueTypeFonts()) {
      fprintf(stderr,
	      "The document contains TrueType fonts.\n"
	      "Ipe cannot save these in Postscript format.\n"
	      "You can save as PDF, and use pdftops to create a "
	      "Postscript file.\n");
    } else if (!doc->Save(dst, "ipetoipe", fm, flags, fromPage, toPage))
      fprintf(stderr, "Failed to save document!\n");
    else if (flags & IpeDocument::EExport)
      fprintf(stderr, "Warning: the exported file contains no Ipe markup!\n");
  }
  return exitCode;
}

static void usage()
{
  fprintf(stderr,
	  "Usage: ipetoipe ( -xml | -eps | -pdf ) [ -export ] "
	  "[ -from <page> ] [ -to <page> ] "
	  "[ -noshading ] [ -lastview ] [ -runlatex ] [ -nocolor ] "
	  "[ -styleupdate ] "
	  "infile outfile\n"
	  "Ipetoipe converts between the different Ipe file formats.\n"
	  " -export   : output contains no Ipe markup.\n"
	  " -from     : start exporting at this page.\n"
	  " -to       : stop exporting at this page.\n"
	  " -lastview : only the last view of each Ipe page is saved.\n"
	  " -runlatex : run Latex even for XML output.\n"
	  " -nocolor  : avoid any color commands in EPS output.\n"
	  " -styleupdate : update style sheets.\n"
	  );
  exit(1);
}

int main(int argc, char *argv[])
{
  // Check Ipelib version mismatch
  if (IpelibVersion() != IPELIB_VERSION) {
    fprintf(stderr,
	    "Ipetoipe has been compiled with header files for Ipelib %d\n"
	    "but is dynamically linked against libipe %d.\n"
	    "Check with 'ldd ipetoipe' which libipe is being loaded, and "
	    "replace it by the correct version or set LD_LIBRARY_PATH.\n",
	    IPELIB_VERSION, IpelibVersion());
    exit(99);
  }

  // set ipeDebug handler
  ipeDebugHandler = ipeDebugImpl;

  // ensure at least three arguments (handles -help as well :-)
  if (argc < 4)
    usage();

  IpeDocument::TFormat frm = IpeDocument::EUnknown;
  if (!qstrcmp(argv[1], "-xml"))
    frm = IpeDocument::EXml;
  else if (!qstrcmp(argv[1], "-eps"))
    frm = IpeDocument::EEps;
  else if (!qstrcmp(argv[1], "-pdf"))
    frm = IpeDocument::EPdf;

  if (frm == IpeDocument::EUnknown)
    usage();

  uint flags = IpeDocument::ESaveNormal;
  bool runLatex = false;
  bool styleUpdate = false;
  int fromPage = -1;
  int toPage = -1;
  int i = 2;
  if (!qstrcmp(argv[i], "-export")) {
    flags |= IpeDocument::EExport;
    ++i;
  }

  if (!qstrcmp(argv[i], "-from")) {
    if (sscanf(argv[i+1], "%d", &fromPage) != 1)
      usage();
    --fromPage;
    i += 2;
  }

  if (!qstrcmp(argv[i], "-to")) {
    if (sscanf(argv[i+1], "%d", &toPage) != 1)
      usage();
    --toPage;
    i += 2;
  }

  if (!qstrcmp(argv[i], "-noshading")) {
    flags |= IpeDocument::ENoShading;
    ++i;
  }

  if (!qstrcmp(argv[i], "-lastview")) {
    flags |= IpeDocument::ELastView;
    ++i;
  }

  if (!qstrcmp(argv[i], "-runlatex")) {
    runLatex = true;
    ++i;
  }

  if (!qstrcmp(argv[i], "-nocolor")) {
    flags |= IpeDocument::ENoColor;
    ++i;
  }

  if (!qstrcmp(argv[i], "-styleupdate")) {
    styleUpdate = true;
    ++i;
  }

  // remaining arguments must be two filenames
  if (argc != i + 2)
    usage();

  if ((flags & IpeDocument::EExport) && frm == IpeDocument::EXml) {
    fprintf(stderr, "-export only available with -pdf and -eps.\n");
    exit(1);
  }

  if ((fromPage >= 0 || toPage >= 0) && !(flags & IpeDocument::EExport)) {
    fprintf(stderr, "-from and -to only available with -export.\n");
    exit(1);
  }

  int reason;
  IpeAutoPtr<IpeDocument> doc(IpeDocument::New(argv[i], reason));

  if (!doc.Ptr()) {
    fprintf(stderr, "Could not read Ipe file '%s' (reason %d)\n",
	    argv[i], reason);
    return 1;
  }

  if (styleUpdate) {
    fprintf(stderr, "%s",
	    IpeModel::UpdateStyleSheets(doc.Ptr(), QDir::current()).CString());
  }

  switch (frm) {
  case IpeDocument::EXml:
    if (runLatex)
      return topdforps(doc.Ptr(), argv[i], argv[i+1],
		       IpeDocument::EXml, flags);
    doc->Save(argv[i+1], "ipetoipe", IpeDocument::EXml,
	      IpeDocument::ESaveNormal);
  default:
    return 0;

  case IpeDocument::EPdf:
    return topdforps(doc.Ptr(), argv[i], argv[i+1], IpeDocument::EPdf,
		     flags, fromPage, toPage);

  case IpeDocument::EEps:
    return topdforps(doc.Ptr(), argv[i], argv[i+1], IpeDocument::EEps, flags);
  }
}

// --------------------------------------------------------------------
