import unittest
import warnings

from markdown_it import MarkdownIt

from iotas.markdown_helpers import (
    parse_to_tokens,
    get_note_export_content,
    get_image_attachments_from_tokens,
    get_image_attachments_from_note_content,
    filter_image_tokens,
    create_attachment_for_markup_path,
    update_image_path_in_markup,
    string_is_image,
)
from iotas.note import Note

warnings.filterwarnings("ignore", "version")


class Test(unittest.TestCase):

    def test_parse_to_tokens(self) -> None:
        note = Note()
        note.content = "plain"
        parser, tokens = parse_to_tokens(note, exporting=False, tex_support=False)
        self.assertIsInstance(parser, MarkdownIt)
        self.assertEqual(len(tokens), 3)

    def test_get_note_export_content(self) -> None:
        note = Note()
        note.content = "![description](image.png)"
        note.id = 1
        out = get_note_export_content(note, prefix_note_id=False)
        self.assertEqual(out, "![description](attachments/image.png)")

        out = get_note_export_content(note, prefix_note_id=True)
        self.assertEqual(out, "![description](attachments/1.image.png)")

    def test_get_image_attachments_from_tokens(self) -> None:
        note = Note()

        note.content = "plain"
        _parser, tokens = parse_to_tokens(note, exporting=False, tex_support=False)
        attachments = get_image_attachments_from_tokens(note, tokens)
        self.assertEqual(len(attachments), 0)

        note.content = "![](image.png)\n\n![](image2.png)"
        _parser, tokens = parse_to_tokens(note, exporting=False, tex_support=False)
        attachments = get_image_attachments_from_tokens(note, tokens)
        self.assertEqual(len(attachments), 2)
        self.assertEqual(attachments[0].path, "image.png")
        self.assertEqual(attachments[1].path, "image2.png")

    def test_get_image_attachments_from_note_content(self) -> None:
        note = Note()

        note.content = "plain"
        attachments = get_image_attachments_from_note_content(note)
        self.assertEqual(len(attachments), 0)

        note.content = "![](image.png)\n\n![](image2.png)"
        attachments = get_image_attachments_from_note_content(note)
        self.assertEqual(len(attachments), 2)
        self.assertEqual(attachments[0].path, "image.png")
        self.assertEqual(attachments[1].path, "image2.png")

        note.content = '![description](image.png "title")'
        attachments = get_image_attachments_from_note_content(note)
        self.assertEqual(len(attachments), 1)
        self.assertEqual(attachments[0].path, "image.png")

    def test_filter_image_tokens(self) -> None:
        note = Note()

        note.content = "plain"
        _parser, tokens = parse_to_tokens(note, exporting=False, tex_support=False)
        tokens = filter_image_tokens(tokens)
        self.assertEqual(len(tokens), 0)

        note.content = "![](image.png)\n\n![](image2.png)"
        _parser, tokens = parse_to_tokens(note, exporting=False, tex_support=False)
        tokens = filter_image_tokens(tokens)
        self.assertEqual(len(tokens), 2)

    def test_create_attachment_for_markup_path(self) -> None:
        note = Note()
        note.id = 1
        note.remote_id = 2
        attachment = create_attachment_for_markup_path(note, "image.png")
        self.assertEqual(attachment.note_id, 1)
        self.assertEqual(attachment.note_remote_id, 2)
        self.assertEqual(attachment.path, "image.png")

    def test_update_image_path_in_markup(self) -> None:
        content = "# basic"
        out = update_image_path_in_markup(content, "old_path.png", "new_path.png")
        self.assertEqual(out, content)

        content = "![description](old.png)"
        out = update_image_path_in_markup(content, "old.png", "new.png")
        self.assertEqual(out, "![description](new.png)")

        content = "![](old.png"
        out = update_image_path_in_markup(content, "old.png", "new.png")
        self.assertEqual(out, "![](old.png")

        content = '![](old.png "title")'
        out = update_image_path_in_markup(content, "old.png", "new.png")
        self.assertEqual(out, '![](new.png "title")')

        content = "[](old.png)"
        out = update_image_path_in_markup(content, "old.png", "new.png")
        self.assertEqual(out, "[](old.png)")

    def test_string_is_image(self) -> None:
        self.assertFalse(string_is_image("plain"))
        self.assertTrue(string_is_image("![](image.png)"))
        self.assertFalse(string_is_image("![](image.png"))
        self.assertFalse(string_is_image("[](image.png)"))
        self.assertFalse(string_is_image(" ![](image.png)"))
        self.assertFalse(string_is_image("![](image.png) "))
        self.assertTrue(string_is_image("![description](image.png)"))
        self.assertTrue(string_is_image('![description](image.png "title")'))
